from __future__ import unicode_literals
import time

from sepelib.core import config

import yp.data_model
from yp_proto.yp.client.api.proto import pod_agent_pb2
from infra.swatlib.gevent import geventutil as gutil
from infra.mc_rsc.src import consts
from infra.mc_rsc.src import pod_id_generator
from infra.mc_rsc.src import yputil

# TODO: remove PCS_START_PENDING from AGENT_ALIVE_STATES
# after fixing https://st.yandex-team.ru/ISS-6019.
AGENT_ALIVE_STATES = (yp.data_model.PCS_STARTED,
                      yp.data_model.PCS_START_PENDING)

EVICTION_ALIVE_STATES = (yp.data_model.ES_NONE, yp.data_model.ES_REQUESTED)
MAINTENANCE_IN_PROGRESS_STATES = (yp.data_model.PMS_ACKNOWLEDGED, yp.data_model.PMS_IN_PROGRESS)
MAINTENANCE_POSSIBLE_KINDS = (yp.data_model.MK_REBOOT, yp.data_model.MK_TEMPORARY_UNREACHABLE, yp.data_model.MK_PROFILE)

YP_TIMESTAMP_MULTIPLIER = 2 ** 30


def is_pod_ready(pod):
    if not pod.status.scheduling.node_id:
        return False
    if pod.status.eviction.state not in EVICTION_ALIVE_STATES:
        return False
    if pod.status.agent.pod_agent_payload.status.revision != pod.spec.pod_agent_payload.spec.revision:
        return False
    if pod.status.agent.state not in AGENT_ALIVE_STATES:
        return False
    if pod.status.agent.pod_agent_payload.status.ready.status == pod_agent_pb2.EConditionStatus_TRUE:
        creation_time_in_seconds = pod.meta.creation_time // 1000000
        ready_period_after_replace = config.get_value('controller.ready_period_after_replace', 0)
        return time.time() - creation_time_in_seconds > ready_period_after_replace
    return False


def is_pod_disabled_permanently(pod):
    d = yputil.get_label(pod.labels, 'deploy', default={})
    return d.get(consts.DISABLE_PERMANENTLY_LABEL, False)


def is_pod_failed(pod):
    return pod.status.agent.pod_agent_payload.status.failed.status == pod_agent_pb2.EConditionStatus_TRUE


def is_pod_eviction_empty(pod):
    return (pod.status.eviction.state == yp.data_model.ES_NONE or
            (pod.status.eviction.state == yp.data_model.ES_REQUESTED and
             pod.status.eviction.reason == yp.data_model.ER_HFSM)
            )


def is_pod_eviction_requested_ignore_reason(pod):
    return pod.status.eviction.state == yp.data_model.ES_REQUESTED


def is_pod_eviction_requested(pod):
    return (pod.status.eviction.state == yp.data_model.ES_REQUESTED and
            pod.status.eviction.reason != yp.data_model.ER_HFSM)


def is_pod_marked_removing_delegate(pod):
    return yputil.get_label(pod.labels, consts.DELEGATE_REMOVING_LABEL, default=False)


def is_pod_eviction_acknowledged(pod):
    return pod.status.eviction.state == yp.data_model.ES_ACKNOWLEDGED


def is_pod_node_alerted(pod):
    disruptive_node_alerts = config.get_value('controller.disruptive_node_alerts', {})
    now = time.time()
    for node_alert in pod.status.node_alerts:
        timeout = disruptive_node_alerts.get(node_alert.type)
        if timeout is None:
            continue
        if now - node_alert.creation_time.seconds < timeout:
            continue
        return True
    return False


def is_pod_graceful_shutdown_required(pod):
    if pod.status.agent_spec_timestamp == 0:
        return False
    return True


def is_maintenance_empty(pod):
    return pod.status.maintenance.state == yp.data_model.PMS_NONE


def is_maintenance_in_progress(pod):
    return pod.status.maintenance.state in MAINTENANCE_IN_PROGRESS_STATES


def is_maintenance_requested(pod):
    return pod.status.maintenance.state == yp.data_model.PMS_REQUESTED


def is_maintenance_overtimed(pod, max_tolerable_downtime_seconds):
    if not max_tolerable_downtime_seconds:
        return False
    maintenance_last_updated = pod.status.maintenance.last_updated / 1000000
    if maintenance_last_updated + max_tolerable_downtime_seconds > time.time():
        return False
    return is_maintenance_in_progress(pod)


def is_maintenance_disruptive(pod, max_tolerable_downtime_seconds):
    if pod.status.maintenance.info.disruptive:
        return True
    if pod.status.maintenance.info.kind not in MAINTENANCE_POSSIBLE_KINDS:
        return True
    return pod.status.maintenance.info.estimated_duration.seconds > max_tolerable_downtime_seconds


def is_workload_updateable(src, target):
    # NOTE: maybe it would be better to compare max init resources
    for s, t in zip(src.init, target.init):
        if s.compute_resources != t.compute_resources:
            return False
    return src.start.compute_resources == target.start.compute_resources


def is_box_updateable(src, target):
    # NOTE: maybe it would be better to compare max init resources
    for s, t in zip(src.init, target.init):
        if s.compute_resources != t.compute_resources:
            return False
    return True


def is_resource_requests_updateable(spec, template):
    # DEPLOY-1722: we must compare every single field
    # to avoid case when YP sets some field value on pod saving
    if spec.vcpu_guarantee != template.vcpu_guarantee:
        return False

    if spec.memory_guarantee != template.memory_guarantee:
        return False
    if spec.memory_limit != template.memory_limit:
        return False

    if spec.network_bandwidth_guarantee != template.network_bandwidth_guarantee:
        return False
    if spec.network_bandwidth_limit != template.network_bandwidth_limit:
        return False
    return True


def is_pod_spec_updateable(spec, template, allow_resources=False):
    if not allow_resources and not is_resource_requests_updateable(spec.resource_requests, template.resource_requests):
        return False
    if spec.disk_volume_requests != template.disk_volume_requests:
        return False
    if spec.ip6_address_requests != template.ip6_address_requests:
        return False
    if not allow_resources and spec.host_devices != template.host_devices:
        return False
    return True


def update_pod_set_needed(ps, template):
    if ps.spec.account_id != template.spec.account_id:
        return True
    if ps.spec.antiaffinity_constraints != template.spec.antiaffinity_constraints:
        return True
    if ps.meta.acl != template.meta.acl:
        return True
    return False


def make_default_access_control_entry(users):
    rv = yp.data_model.TAccessControlEntry()
    rv.action = yp.data_model.ACA_ALLOW
    rv.permissions.extend([yp.data_model.ACP_READ,
                           yp.data_model.ACA_WRITE,
                           yp.data_model.ACA_CREATE,
                           yp.data_model.ACA_SSH_ACCESS,
                           yp.data_model.ACA_ROOT_SSH_ACCESS])
    rv.subjects.extend(users)
    return rv


def make_pod_set(mc_rs, default_acl, labels, cluster):
    ps = yp.data_model.TPodSet()
    ps.labels.MergeFrom(labels)

    ps.meta.id = mc_rs.make_ps_id()
    ps.meta.acl.MergeFrom(default_acl)
    ps.meta.acl.MergeFrom(mc_rs.meta.acl)

    spec = mc_rs.spec
    ps.spec.account_id = spec.account_id
    node_segment_label = yputil.get_label(mc_rs.spec.pod_template_spec.labels,
                                          consts.TEMP_NODE_SEGMENT_LABEL)
    ps.spec.node_segment_id = (node_segment_label or
                               spec.node_segment_id or
                               consts.DEFAULT_NODE_SEGMENT_ID)
    ps.spec.antiaffinity_constraints.MergeFrom(
        mc_rs.constraints(cluster).antiaffinity_constraints
    )
    return ps


def is_enumerated_id_allowed(pod_template):
    is_enabled = yputil.get_label(pod_template.labels, consts.ENABLE_ENUMERATED_POD_IDS_LABEL)
    return is_enabled and consts.ENUMERATED_POD_SET_ID_REGEX.match(pod_template.meta.pod_set_id) is not None


def make_pod_template(mc_rs, match_labels):
    rv = yp.data_model.TPod()
    rv.meta.pod_set_id = mc_rs.make_ps_id()
    tpl = mc_rs.spec.pod_template_spec
    rv.spec.CopyFrom(tpl.spec)
    if not yputil.get_label(tpl.labels, consts.DISABLE_PODS_MOVE_LABEL):
        rv.spec.enable_scheduling = True
    rv.labels.CopyFrom(tpl.labels)
    rv.labels.MergeFrom(match_labels)
    rv.annotations.CopyFrom(tpl.annotations)
    return rv


def copy_allocation(old_pod_spec, new_pod_spec):
    new_pod_spec.resource_requests.vcpu_guarantee = old_pod_spec.resource_requests.vcpu_guarantee
    new_pod_spec.resource_requests.memory_guarantee = old_pod_spec.resource_requests.memory_guarantee
    new_pod_spec.resource_requests.memory_limit = old_pod_spec.resource_requests.memory_limit
    new_pod_spec.resource_requests.network_bandwidth_guarantee = old_pod_spec.resource_requests.network_bandwidth_guarantee
    new_pod_spec.resource_requests.network_bandwidth_limit = old_pod_spec.resource_requests.network_bandwidth_limit

    del new_pod_spec.disk_volume_requests[:]
    new_pod_spec.disk_volume_requests.extend(old_pod_spec.disk_volume_requests)

    del new_pod_spec.ip6_address_requests[:]
    new_pod_spec.ip6_address_requests.extend(old_pod_spec.ip6_address_requests)

    del new_pod_spec.host_devices[:]
    new_pod_spec.host_devices.extend(old_pod_spec.host_devices)


def make_pod(pod_id, pod_set_id, spec, labels, annotations):
    p = yp.data_model.TPod()
    p.meta.id = pod_id
    p.meta.inherit_acl = True
    p.meta.pod_set_id = pod_set_id
    p.spec.CopyFrom(spec)
    p.spec.pod_agent_payload.spec.id = pod_id
    p.labels.CopyFrom(labels)
    p.annotations.CopyFrom(annotations)
    return p


def make_pod_with_index(pod_id, pod_index, pod_set_id, spec, labels, annotations):
    p = make_pod(pod_id, pod_set_id, spec, labels, annotations)
    yputil.set_label(p.labels, consts.POD_INDEX_LABEL, pod_index)
    return p


def make_random_id_pods(pod_template, count):
    gen = pod_id_generator.RANDOM_POD_ID_GENERATOR
    pods = []
    for _ in gutil.gevent_idle_iter(xrange(count)):
        p_id = gen.generate()
        p = make_pod(pod_id=p_id,
                     pod_set_id=pod_template.meta.pod_set_id,
                     spec=pod_template.spec,
                     labels=pod_template.labels,
                     annotations=pod_template.annotations)
        pods.append(p)
    return pods


def make_random_id_pods_by_other_pods(pod_template, pod_ids):
    pods = []
    for p_id in gutil.gevent_idle_iter(pod_ids):
        p = make_pod(pod_id=p_id,
                     pod_set_id=pod_template.meta.pod_set_id,
                     spec=pod_template.spec,
                     labels=pod_template.labels,
                     annotations=pod_template.annotations)
        pods.append(p)
    return pods


def make_random_id_pods_by_other_pods_preserve_allocation(pod_template, pod_ids, pod_storage, cluster):
    pods = []
    for p_id in gutil.gevent_idle_iter(pod_ids):
        p = pod_storage.get(p_id, cluster)
        if not p:
            continue
        copy_allocation(p.spec, pod_template.spec)
        p = make_pod(pod_id=p_id,
                     pod_set_id=pod_template.meta.pod_set_id,
                     spec=pod_template.spec,
                     labels=pod_template.labels,
                     annotations=pod_template.annotations)
        pods.append(p)
    return pods


def make_enumerated_id_pods(pod_template, count, pod_storage, cluster):
    ps_id = pod_template.meta.pod_set_id
    gen = pod_id_generator.EnumeratedPodIdGenerator(pod_set_id=ps_id,
                                                    pod_storage=pod_storage,
                                                    cluster=cluster)
    pods = []
    for _ in gutil.gevent_idle_iter(xrange(count)):
        p_id = gen.generate()
        p = make_pod_with_index(pod_id=p_id.id,
                                pod_index=p_id.index,
                                pod_set_id=ps_id,
                                spec=pod_template.spec,
                                labels=pod_template.labels,
                                annotations=pod_template.annotations)
        pods.append(p)
    return pods


def make_enumerated_id_pods_by_other_pods(pod_template, pod_ids, pod_storage,
                                          cluster):
    pods = []
    for p_id in gutil.gevent_idle_iter(pod_ids):
        p = pod_storage.get(p_id, cluster)
        if not p:
            continue

        p_idx = yputil.get_label(p.labels, consts.POD_INDEX_LABEL)
        p = make_pod_with_index(pod_id=p.meta.id,
                                pod_index=p_idx,
                                pod_set_id=pod_template.meta.pod_set_id,
                                spec=pod_template.spec,
                                labels=pod_template.labels,
                                annotations=pod_template.annotations)
        pods.append(p)
    return pods


def make_enumerated_id_pods_by_other_pods_preserve_allocation(pod_template, pod_ids, pod_storage,
                                                              cluster):
    pods = []
    for p_id in gutil.gevent_idle_iter(pod_ids):
        p = pod_storage.get(p_id, cluster)
        if not p:
            continue
        copy_allocation(p.spec, pod_template.spec)
        p_idx = yputil.get_label(p.labels, consts.POD_INDEX_LABEL)
        p = make_pod_with_index(pod_id=p.meta.id,
                                pod_index=p_idx,
                                pod_set_id=pod_template.meta.pod_set_id,
                                spec=pod_template.spec,
                                labels=pod_template.labels,
                                annotations=pod_template.annotations)
        pods.append(p)
    return pods


def make_pods(pod_template, count, pod_storage, cluster):
    if is_enumerated_id_allowed(pod_template):
        return make_enumerated_id_pods(pod_template=pod_template,
                                       count=count,
                                       pod_storage=pod_storage,
                                       cluster=cluster)
    return make_random_id_pods(pod_template=pod_template,
                               count=count)


def make_pods_by_other_pods(pod_template, pod_ids, pod_storage, cluster):
    if is_enumerated_id_allowed(pod_template):
        return make_enumerated_id_pods_by_other_pods(pod_template=pod_template,
                                                     pod_ids=pod_ids,
                                                     pod_storage=pod_storage,
                                                     cluster=cluster)
    return make_random_id_pods_by_other_pods(pod_template=pod_template,
                                             pod_ids=pod_ids)


def make_pods_by_other_pods_preserve_allocation(pod_template, pod_ids, pod_storage, cluster):
    if is_enumerated_id_allowed(pod_template):
        return make_enumerated_id_pods_by_other_pods_preserve_allocation(pod_template=pod_template,
                                                                         pod_ids=pod_ids,
                                                                         pod_storage=pod_storage,
                                                                         cluster=cluster)
    return make_random_id_pods_by_other_pods_preserve_allocation(pod_template=pod_template,
                                                                 pod_ids=pod_ids,
                                                                 pod_storage=pod_storage,
                                                                 cluster=cluster)


def is_target_state_removed(pod):
    return pod.spec.pod_agent_payload.spec.target_state == yp.data_model.EPodAgentTargetState_REMOVED


def is_spec_applied_by_pod_agent(pod, spec_change_timestamp):
    return spec_change_timestamp <= pod.status.agent.pod_agent_payload.status.spec_timestamp


def is_removed_ready(pod):
    target_removed = is_target_state_removed(pod)
    current_spec_applied = is_spec_applied_by_pod_agent(pod, pod.status.master_spec_timestamp)
    return target_removed and current_spec_applied and is_pod_ready(pod)


def is_destroy_overtimed(pod):
    if not is_target_state_removed(pod):
        return False
    if is_pod_ready(pod) and is_spec_applied_by_pod_agent(pod, pod.status.master_spec_timestamp):
        return False

    max_timeout = config.get_value('controller.max_destroy_hook_period_secs',
                                   consts.MAX_DESTROY_HOOK_PERIOD_SECS)
    update_timestamp = pod.status.master_spec_timestamp // YP_TIMESTAMP_MULTIPLIER
    return update_timestamp + max_timeout < time.time()
