from __future__ import unicode_literals
import logging
import datetime

import gevent
from infra.swatlib.gevent import greenthread
from infra.swatlib.gevent import geventutil as gutil
from infra.mc_rsc.src import reflector
from infra.mc_rsc.src import sync_status


def make_cluster_failed_condition(cluster):
    c = sync_status.SyncCondition(cluster)
    c.set_error(
        reason=reflector.CLUSTER_UNAVAILABLE_REASON,
        message=reflector.CLUSTER_UNAVAILABLE_MESSAGE_TPL.format(c)
    )
    return c


class Runner(greenthread.GreenThread):

    def __init__(self, ctl, deploy_engine, xdc,
                 yp_clients,
                 mc_rs_storage, circuit_breaker,
                 pod_reflectors, ps_reflectors,
                 mc_rs_reflector, relation_reflector,
                 threads_count,
                 relation_sync_delay,
                 allowed_mc_rs_ids, ignored_mc_rs_ids,
                 metrics_registry, metrics_collector):

        super(Runner, self).__init__()
        self.log = logging.getLogger('ctl_runner')
        self.ctl = ctl
        self.deploy_engine = deploy_engine
        self.xdc = xdc
        self.yp_clients = yp_clients
        self.mc_rs_storage = mc_rs_storage
        self.circuit_breaker = circuit_breaker
        self.pod_reflectors = pod_reflectors
        self.ps_reflectors = ps_reflectors
        self.mc_rs_reflector = mc_rs_reflector
        self.relation_reflector = relation_reflector
        self.last_run_time = None
        self.allowed_mc_rs_ids = allowed_mc_rs_ids
        self.ignored_mc_rs_ids = ignored_mc_rs_ids
        self._metrics_registry = metrics_registry
        self._metrics_collector = metrics_collector
        self._gs = []  # List of active controller greenlets
        self._pool = gevent.pool.Pool(threads_count)
        m = 'run_time_{}_{}'.format(self.xdc, self.deploy_engine.lower())
        self._run_time_gauge = self._metrics_registry.get_gauge(m)
        self.relations_sync_last_time = None
        self.relation_sync_delay = relation_sync_delay

    def _handle_ctl_exception(self, exc, mc_rs_id):
        self.log.exception("controller for mc_rs %s finished with error: %s",
                           mc_rs_id, str(exc))

    def is_relation_sync_allowed(self):
        if self.relations_sync_last_time is None:
            return True
        d = datetime.timedelta(seconds=self.relation_sync_delay)
        return datetime.datetime.now() - self.relations_sync_last_time >= d

    def start_ctl(self, mc_rs, failed_clusters):
        mc_rs_id = mc_rs.meta.id
        self.log.info("starting controller for mc_rs %s...", mc_rs_id)
        g = gevent.Greenlet(self.ctl.process, mc_rs, failed_clusters)
        g.link_exception(lambda e: self._handle_ctl_exception(e, mc_rs_id))
        self._gs.append(g)
        self._pool.start(g)
        self.log.info("started controller for mc_rs %s", mc_rs_id)

    def collect_ctl_metrics(self):
        results = []
        for g in self._gs:
            if not g.value:
                continue
            if isinstance(g.value, gevent.GreenletExit):
                continue
            results.append(g.value)
        self._metrics_collector.collect(results)

    def is_mc_rs_allowed(self, mc_rs):
        if self.xdc in mc_rs.disabled_clusters():
            self.log.info("skip starting controller for %s, it is disabled in /labels/deploy/disabled_clusters", mc_rs.meta.id)
            return False
        if self.allowed_mc_rs_ids:
            return mc_rs.meta.id in self.allowed_mc_rs_ids
        return mc_rs.meta.id not in self.ignored_mc_rs_ids

    def start_and_wait_ctls(self, failed_clusters):
        m = 'iteration_time_{}_{}'.format(self.xdc,
                                          self.deploy_engine.lower())
        with self._metrics_registry.get_gauge(m).timer() as t:
            for mc_rs in self.mc_rs_storage.list():
                if self.is_mc_rs_allowed(mc_rs):
                    self.start_ctl(mc_rs, failed_clusters)
            gevent.wait(self._gs)
            try:
                self.collect_ctl_metrics()
            except Exception:
                self.log.exception("controller metrics collection is failed")
            del self._gs[:]

        self.log.info("iteration_time %s", t.gauge.get())

    def start_and_wait_reflectors(self):
        self.log.info('starting reflectors...')
        threads = []
        results = []
        try:
            yp_xdc_client = self.yp_clients[self.xdc]
            try:
                ts = yp_xdc_client.generate_timestamp()
            except Exception:
                self.log.exception("failed generation timestamp in %s: restarting controller loop", self.xdc)
                return [make_cluster_failed_condition(self.xdc)]
            threads.append(gevent.spawn(self.mc_rs_reflector.start, timestamp=ts))
            if self.is_relation_sync_allowed():
                threads.append(gevent.spawn(self.relation_reflector.start, timestamp=ts))
                self.relations_sync_last_time = datetime.datetime.now()
            self.log.info("relation_storage size: %d", self.relation_reflector.storage.size())
            for cluster, client in self.yp_clients.iteritems():
                try:
                    ts = client.generate_timestamp()
                except Exception:
                    self.log.exception("failed generation timestamp in %s: skipping cluster", cluster)
                    results.append(make_cluster_failed_condition(cluster))
                    continue
                pod_reflector = self.pod_reflectors.get(cluster)
                if pod_reflector:
                    threads.append(gevent.spawn(pod_reflector.start, timestamp=ts))
                ps_reflector = self.ps_reflectors.get(cluster)
                if ps_reflector:
                    threads.append(gevent.spawn(ps_reflector.start, timestamp=ts))
            gevent.joinall(threads, raise_error=True)
            for t in threads:
                results.append(t.get())
        finally:
            self.log.info('stopping all reflectors...')
            for t in threads:
                gutil.force_kill_greenlet(t, ignore_greenlet_exit=True,
                                          log=self.log)
        return results

    def sync_storages(self):
        self.log.info('syncing storages...')
        sync_conditions = self.start_and_wait_reflectors()
        xdc_error = None
        cluster_errors = {}
        for sync_condition in sync_conditions:
            if sync_condition.condition.succeeded:
                continue
            if sync_condition.cluster == self.xdc:
                xdc_error = sync_condition.condition
            else:
                cluster_errors[sync_condition.cluster] = sync_condition.condition
        return xdc_error, cluster_errors

    def run_once(self):
        """
        Run one iteration with time measurement.
        """
        t = self._run_time_gauge.timer()
        try:
            self._run_once()
        finally:
            t.stop()

    def _run_once(self):
        try:
            xdc_error, cluster_errors = self.sync_storages()
        except Exception:
            self.log.exception('syncing storages failed with unexpected error')
            return

        if xdc_error:
            self.log.error('failed to sync cluster: %s: %s: '
                           'replica set storage is not synced, '
                           'controllers will not be started',
                           self.xdc, xdc_error)
            return
        if not cluster_errors:
            self.log.info('synced storages successfully')
        else:
            self.log.error('synced storages with errors: '
                           'failed to sync clusters: %s: '
                           'controllers will be started on failed clusters',
                           ', '.join(cluster_errors.iterkeys()))

        self.log.info('starting controllers...')
        self.start_and_wait_ctls(failed_clusters=cluster_errors)
        self.circuit_breaker.reset()

    def run(self):
        try:
            while True:
                self.run_once()
        finally:
            self.log.info("stopping all controllers...")
            for g in self._gs:
                gutil.force_kill_greenlet(g, ignore_greenlet_exit=True,
                                          log=self.log)
            del self._gs[:]
            self.log.info("stopped all controllers")
