from __future__ import unicode_literals
import itertools
import math
import yp.data_model
from google.protobuf import timestamp_pb2

from infra.swatlib.gevent import geventutil as gutil
from infra.mc_rsc.src import yputil
from consts import ReadyThresholdTypes


class ReplicaSetStatusMaker(object):

    @staticmethod
    def get_deploy_status_by_cluster(status, cluster):
        return status.deploy_status

    @staticmethod
    def get_current_revision_progress(status):
        return status.deploy_status.details.current_revision_progress

    @staticmethod
    def get_ready(status):
        return status.ready_condition

    @staticmethod
    def get_in_progress(status):
        return status.in_progress_condition

    @staticmethod
    def fill_condition_timestamp(new_condition, old_condition):
        is_equal = (new_condition.status == old_condition.status and
                    new_condition.reason == old_condition.reason and
                    new_condition.message == old_condition.message)
        if is_equal:
            new_condition.last_transition_time.CopyFrom(
                old_condition.last_transition_time
            )
        else:
            ts = timestamp_pb2.Timestamp()
            ts.GetCurrentTime()
            new_condition.last_transition_time.CopyFrom(ts)

    @staticmethod
    def increment_progress(p, ready=0, in_progress=0, failed=0):
        p.pods_ready += ready
        p.pods_in_progress += in_progress
        p.pods_failed += failed
        p.pods_total += ready + in_progress + failed

    @classmethod
    def increment_revision_progress(cls, details,
                                    revision, target,
                                    ready, in_progress, failed):
        """
        :type details: yp_proto.yp.client.api.proto.deploy_pb2.TDeployStatusDetails
        """
        cls.increment_progress(details.revisions[revision],
                               ready, in_progress, failed)
        cls.increment_progress(details.total_progress,
                               ready, in_progress, failed)
        if revision == target:
            cls.increment_progress(details.current_revision_progress,
                                   ready, in_progress, failed)

    @staticmethod
    def fill_status_revision(status, target):
        status.revision_id = str(target)

    @classmethod
    def fill_deploy_status_details(cls, new_details, old_details,
                                   current_state, target, replica_count,
                                   ctl_error, failed_cluster):
        for r in gutil.gevent_idle_iter(current_state.ready.revisions()):
            c = current_state.ready.count(r)
            cls.increment_revision_progress(new_details, r, target,
                                            ready=c,
                                            in_progress=0,
                                            failed=0)

        for r in gutil.gevent_idle_iter(current_state.in_progress.revisions()):
            c = current_state.in_progress.count(r)
            cls.increment_revision_progress(new_details, r, target,
                                            ready=0,
                                            in_progress=c,
                                            failed=0)

        for r in gutil.gevent_idle_iter(current_state.target_state_removed.revisions()):
            # Consider target_state=removed as in_progress
            c = current_state.target_state_removed.count(r)
            cls.increment_revision_progress(new_details, r, target,
                                            ready=0,
                                            in_progress=c,
                                            failed=0)

        for r in gutil.gevent_idle_iter(current_state.failed.revisions()):
            c = current_state.failed.count(r)
            cls.increment_revision_progress(new_details, r, target,
                                            ready=0,
                                            in_progress=0,
                                            failed=c)
        # DEPLOY-5307: fill new mapped_revisions with yson_map annotation.
        # revisions field is legacy one.
        for r, v in new_details.revisions.iteritems():
            new_details.mapped_revisions[str(r)].CopyFrom(v)

        new_details.current_revision_progress.pods_total = replica_count

        succeeded = new_details.controller_status.last_attempt.succeeded
        succeeded.last_transition_time.GetCurrentTime()
        if failed_cluster:
            succeeded.status = yp.data_model.CS_FALSE
            succeeded.message = failed_cluster.message
            succeeded.reason = failed_cluster.reason
        elif ctl_error:
            succeeded.status = yp.data_model.CS_FALSE
            succeeded.message = succeeded.reason = str(ctl_error)
        else:
            succeeded.status = yp.data_model.CS_TRUE
            succeeded.message = succeeded.reason = ''

        cls.fill_condition_timestamp(
            new_condition=succeeded,
            old_condition=old_details.controller_status.last_attempt.succeeded
        )

    @classmethod
    def has_old_revision_pods(cls, mc_rs, current_state):
        target = mc_rs.spec_revision()
        old_revision_pods = itertools.chain(current_state.in_progress.exclude(target),
                                            current_state.ready.exclude(target),
                                            current_state.failed.exclude(target))
        return bool(list(old_revision_pods))

    @staticmethod
    def _is_ready_threshold_achieved(ready_threshold, max_unavailable, replica_count, ready_target_pods_count):
        if replica_count == 0:
            return True
        if ready_threshold.type in ("", ReadyThresholdTypes.MAX_UNAVAILABLE):
            # by default or for MAX_UNAVAILABLE require at least 1 ready pod
            min_ready = max(1, replica_count - max_unavailable)
        elif ready_threshold.type == ReadyThresholdTypes.AUTO:
            min_ready = replica_count - int(math.log(replica_count, 3))
        elif ready_threshold.type == ReadyThresholdTypes.MIN_READY_PODS_PERCENT:
            min_ready_percent = float(ready_threshold.min_ready_pods_percent.value) / 100
            min_ready = math.ceil(min_ready_percent * replica_count)
        elif ready_threshold.type == ReadyThresholdTypes.MAX_NOT_READY_PODS:
            min_ready = replica_count - ready_threshold.max_not_ready_pods.value
        else:
            raise ValueError('Unknown type {} of ready_threshold,'
                             'expected one of ("", "AUTO", "MAX_UNAVAILABLE", "MIN_READY_PODS_PERCENT", "MAX_NOT_READY_PODS")').format(ready_threshold.type)
        min_ready = min(replica_count, min_ready)
        return ready_target_pods_count >= min_ready

    @classmethod
    def _process_ready_criterion(cls, mc_rs, current_state):
        ready_criterion = mc_rs.spec.deployment_strategy.ready_criterion
        if cls.has_old_revision_pods(mc_rs, current_state):
            return False
        max_unavailable = mc_rs.spec.deployment_strategy.max_unavailable
        ready_target_pods_count = current_state.ready.count(mc_rs.spec_revision())
        return cls._is_ready_threshold_achieved(ready_criterion.ready_threshold, max_unavailable,
                                                mc_rs.replica_count(), ready_target_pods_count)

    @classmethod
    def fill_status_ready_in_progress(cls, status, mc_rs, current_state):
        is_ready = cls._process_ready_criterion(mc_rs, current_state)

        ready = cls.get_ready(status)
        in_progress = cls.get_in_progress(status)
        if is_ready:
            ready.status = yp.data_model.CS_TRUE
            in_progress.status = yp.data_model.CS_FALSE
        else:
            ready.status = yp.data_model.CS_FALSE
            in_progress.status = yp.data_model.CS_TRUE

        cls.fill_condition_timestamp(
            old_condition=cls.get_ready(mc_rs.status),
            new_condition=cls.get_ready(status)
        )
        cls.fill_condition_timestamp(
            old_condition=cls.get_in_progress(mc_rs.status),
            new_condition=cls.get_in_progress(status)
        )

    @classmethod
    def fill_cluster_deploy_statuses(cls, status, mc_rs,
                                     current_state, target,
                                     ctl_errors, failed_clusters):
        for c in mc_rs.list_spec_clusters():
            s = current_state.make_filtered_by_cluster_current_state(c)
            deploy_status = cls.get_deploy_status_by_cluster(status, c)
            ctl_error = ctl_errors.get(c)
            failed_cluster = failed_clusters.get(c)
            old_details = cls.get_deploy_status_by_cluster(mc_rs.status, c).details
            cls.fill_deploy_status_details(
                new_details=deploy_status.details,
                old_details=old_details,
                current_state=s,
                target=target,
                replica_count=mc_rs.replica_count_by_cluster(c),
                ctl_error=ctl_error,
                failed_cluster=failed_cluster
            )
            deploy_status.pod_set_id = mc_rs.make_ps_id()

    def make_status(self, mc_rs, current_state, ctl_errors, failed_clusters):
        for e in ctl_errors.itervalues():
            yputil.clear_nondeterministic_attrs_from_yt_error(e)
        status = yp.data_model.TReplicaSetStatus()
        target = mc_rs.spec_revision()
        self.fill_status_revision(status, target)
        self.fill_cluster_deploy_statuses(status=status,
                                          mc_rs=mc_rs,
                                          current_state=current_state,
                                          target=target,
                                          ctl_errors=ctl_errors,
                                          failed_clusters=failed_clusters)
        self.fill_status_ready_in_progress(status=status,
                                           mc_rs=mc_rs,
                                           current_state=current_state)
        return status, self.get_ready(status).status == yp.data_model.CS_TRUE


class MultiClusterReplicaSetStatusMaker(ReplicaSetStatusMaker):

    @staticmethod
    def get_deploy_status_by_cluster(status, cluster):
        return status.cluster_deploy_statuses[cluster]

    @staticmethod
    def get_current_revision_progress(status):
        return status.multi_cluster_deploy_status.details.current_revision_progress

    @staticmethod
    def get_ready(status):
        return status.ready

    @staticmethod
    def get_in_progress(status):
        return status.in_progress

    @staticmethod
    def fill_status_revision(status, target):
        status.revision = target

    @classmethod
    def fill_multi_cluster_deploy_status(cls, status, mc_rs,
                                         current_state, target,
                                         ctl_errors, failed_clusters):
        ctl_error = None
        if ctl_errors:
            ctl_error = ctl_errors.values()[0]
        failed_cluster = None
        if failed_clusters:
            failed_cluster = failed_clusters.values()[0]
        multi_cluster_details = status.multi_cluster_deploy_status.details
        cls.fill_deploy_status_details(
            new_details=multi_cluster_details,
            old_details=mc_rs.status.multi_cluster_deploy_status.details,
            current_state=current_state,
            target=target,
            replica_count=mc_rs.replica_count(),
            ctl_error=ctl_error,
            failed_cluster=failed_cluster
        )

    def make_status(self, mc_rs, current_state, ctl_errors, failed_clusters):
        for e in ctl_errors.itervalues():
            yputil.clear_nondeterministic_attrs_from_yt_error(e)
        status = yp.data_model.TMultiClusterReplicaSetStatus()
        target = mc_rs.spec_revision()
        self.fill_status_revision(status, target)
        self.fill_cluster_deploy_statuses(status=status,
                                          mc_rs=mc_rs,
                                          current_state=current_state,
                                          target=target,
                                          ctl_errors=ctl_errors,
                                          failed_clusters=failed_clusters)
        # DEPLOY-5307: fill new mapped_cluster_deploy_statuses with yson_map
        # annotation. cluster_deploy_statuses field is legacy one.
        status.mapped_cluster_deploy_statuses.MergeFrom(status.cluster_deploy_statuses)
        self.fill_multi_cluster_deploy_status(status=status,
                                              mc_rs=mc_rs,
                                              current_state=current_state,
                                              target=target,
                                              ctl_errors=ctl_errors,
                                              failed_clusters=failed_clusters)
        self.fill_status_ready_in_progress(status=status,
                                           mc_rs=mc_rs,
                                           current_state=current_state)
        return status, self.get_ready(status).status == yp.data_model.CS_TRUE
