#include "application.h"

#include <library/cpp/logger/filter.h>
#include <library/cpp/logger/rotating_file.h>
#include <library/cpp/logger/stream.h>

namespace NMonitoring {
    const TString HTTP_PORT_OPTION = "http-port";
    const TString HTTP_THREADS_OPTION = "http-threads";

    void FillParser(NLastGetopt::TOpts& options) {
        options.AddLongOption(HTTP_PORT_OPTION,
                              "http server port to listen.")
            .Optional()
            .RequiredArgument()
            .DefaultValue("17231");
        options.AddLongOption(HTTP_THREADS_OPTION,
                              "number of threads to process incoming requests")
            .Optional();
        options.AddLongOption('c', "config", "Path to config file.").Optional().RequiredArgument().DefaultValue("");
        options.AddLongOption("log-file", "Path to log file.").Optional().RequiredArgument().DefaultValue("console");
        options.AddLongOption("verbose", "Enable verbose mode.").Optional().HasArg(NLastGetopt::NO_ARGUMENT);
        options.AddLongOption("rotate-log", "Rotate logs.").Optional().HasArg(NLastGetopt::NO_ARGUMENT);
        options.AddLongOption("json-logs", "Json log format").Optional().HasArg(NLastGetopt::NO_ARGUMENT);

        options.AddHelpOption('h');
        options.AddVersionOption('v');
    }

    void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed, TSettings& settings) {
        if (parsed.Has("config")) {
            try {
                const TFile configFile(parsed.Get("config"),
                                       EOpenModeFlag::OpenExisting | EOpenModeFlag::RdOnly | EOpenModeFlag::Seq);
                TFileInput configFileBuffer(configFile);
                settings.FromJson(configFileBuffer);
            } catch (...) {
                throw yexception() << "Can't parse config: " << CurrentExceptionMessage();
            }
        }

        auto httpPort = FromString<ui16>(parsed.Get(HTTP_PORT_OPTION));
        settings.SetHttpPort(httpPort);

        // Check for optional value. Note: optional option value (if present) will override the one from file config.
        if (parsed.Has(HTTP_THREADS_OPTION)) {
            auto httpThreads = FromString<ui16>(parsed.Get(HTTP_THREADS_OPTION));
            settings.SetHttpThreadPoolSize(httpThreads);
        }
    }

    THolder<TLogBackend> CreateRotatingLogBackend(const TString& logType, ELogPriority priority, bool rotate) {
        if (logType == "console") {
            return THolder<TLogBackend>(new TFilteredLogBackend(MakeHolder<TStreamLogBackend>(&Cerr), priority));
        } else if (rotate) {
            return THolder<TLogBackend>(
                new TFilteredLogBackend(
                    MakeHolder<TRotatingFileLogBackend>(logType, 64 * 1024 * 1024, 1), priority));
        } else {
            return THolder<TLogBackend>(CreateLogBackend(logType, priority).Release());
        }
    }
} // namespace NMonitoring
