#pragma once

#include "logging.h"
#include "metrics.h"
#include "settings.h"
#include "web_handlers.h"
#include "metrics_replier.h"
#include "web_server.h"

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/logger/global/global.h>

#include <util/generic/singleton.h>
#include <util/stream/file.h>
#include <util/stream/input.h>
#include <util/string/builder.h>
#include <util/system/file.h>

namespace NMonitoring {
    class IApplication {
    public:
        virtual void Run() = 0;

        virtual void Setup() = 0;

        virtual void Stop(int signal) = 0;

        virtual void ReopenLog(int signal) = 0;

        virtual THolder<TStatsInitializer> GetStatsInitializer() = 0;

        virtual ~IApplication() {
        }
    };

    class TDaemon: public TNonCopyable {
        Y_DECLARE_SINGLETON_FRIEND()

    public:
        ~TDaemon() {
            App = nullptr;
            Logger = nullptr;
        }

        void SetLoggerInstance(THolder<TLog>&& logger) {
            Logger.Reset(logger);
        }

        TLog* GetLogger() {
            return Logger.Get();
        }

        void SetAppInstance(THolder<IApplication>&& app) {
            App.Reset(app);
        }

        IApplication* GetApp() {
            return App.Get();
        }

        void Stop(int signal) {
            if (App != nullptr) {
                App->Stop(signal);
            }
        }

        void ReopenLog(int signal) {
            if (App != nullptr) {
                App->ReopenLog(signal);
            }
        }

        static TDaemon* Get() {
            return SingletonWithPriority<TDaemon, 110000>();
        }

    private:
        TDaemon() {
            signal(SIGINT, StopAppStatic);
            signal(SIGTERM, StopAppStatic);
            signal(SIGHUP, ReopenLogStatic);
        }

        static void StopAppStatic(int sig) noexcept {
            TDaemon::Get()->Stop(sig);
        }

        static void ReopenLogStatic(int sig) noexcept {
            TDaemon::Get()->ReopenLog(sig);
        }

        THolder<TLog> Logger = nullptr;
        THolder<IApplication> App = nullptr;
    };

    template <class TSettingsType>
    class TBaseApplication: public IApplication {
    public:
        TBaseApplication(TLog& logger, const TSettingsType& settings)
            : Settings(settings)
            , Logger(logger)
            , HttpServer(
                  MakeHttpOptions(settings),
                  Logger)
            , MetricsHandler(NMonitoring::TMetricRegistry::Instance()) {
        }

        TBaseApplication(TLog& logger, const TSettingsType& settings, THolder<IThreadPool> httpPool)
            : Settings(settings)
            , Logger(logger)
            , HttpServer(
                  MakeHttpOptions(Settings),
                  std::move(httpPool),
                  Logger)
            , MetricsHandler(NMonitoring::TMetricRegistry::Instance()) {
        }

        virtual void Setup() override {
            // setup yasm metrics
            GetStatsInitializer()->Init(TUnistat::Instance());
            NMonitoring::TMetricRegistry::Instance()->LazyCounter({{"sensor", "uptimeMillis"}}, [startedAt = TInstant::Now()]() {
                auto uptime = TInstant::Now() - startedAt;
                return uptime.MilliSeconds();
            });

            // setup web server
            Logger << TLOG_INFO << "Server registered on http://*:" << Settings.GetHttpPort();
            HttpServer.Add("/stats", StatsHandler);
            HttpServer.Add("/stats/", StatsHandler);
            HttpServer.Add("/ping", OkHandler);
            HttpServer.Add("/metrics", MetricsHandler);
        }

        virtual void Stop(int signal) override {
            if (signal != 0) {
                Logger << TLOG_INFO << "Stopping application on signal " << signal;
            }
            Event.Signal();
            HttpServer.Stop();
        }

        virtual void ReopenLog(int signal) override {
            Logger.ReopenLog();
            Logger << TLOG_NOTICE << "Reopened log on signal " << signal;
        }

        virtual THolder<TStatsInitializer> GetStatsInitializer() override {
            return MakeHolder<TStatsInitializer>();
        }

        virtual ~TBaseApplication() override {
            Stop(0);
        }

    private:
        static THttpServerOptions MakeHttpOptions(const TSettingsType& settings) {
            return THttpServerOptions()
                .SetPort(settings.GetHttpPort())
                .SetThreads(settings.GetHttpThreadPoolSize())
                .EnableCompression(settings.GetEnableCompression())
                .SetMaxQueueSize(settings.GetMaxQueueSize())
                .SetMaxConnections(settings.GetMaxConnections())
                .SetListenBacklog(settings.GetListenBacklog())
                .SetClientTimeout(settings.GetClientTimeout());
        }

    protected:
        const TSettingsType& Settings;
        TLog& Logger;
        TAutoEvent Event;

        TWebServer HttpServer;
        TStatsHandler StatsHandler;
        TMetricsHandler MetricsHandler;

        TOkHandler OkHandler;
    };

    void FillParser(NLastGetopt::TOpts& options);

    void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed, TSettings& settings);

    THolder<TLogBackend> CreateRotatingLogBackend(const TString& logType, ELogPriority priority, bool rotate = false);

    struct TDefaultLoggerFactory {
        THolder<TLog> MakeLogger(THolder<TLogBackend> logBackend) const {
            return MakeHolder<TLog>(std::move(logBackend));
        }
    };

    struct TThreadedLoggerFactory {
        THolder<TLog> MakeLogger(THolder<TLogBackend> logBackend) const {
            return MakeHolder<TLog>(MakeHolder<TOwningThreadedLogBackend>(logBackend.Release()));
        }
    };

    template <class TApp, class TSettingsType, class TLoggerFactory = TDefaultLoggerFactory>
    int RunFromMain(int argc, const char** argv) {
        auto options = NLastGetopt::TOpts::Default();
        FillParser(options);
        TApp::FillParser(options);

        NLastGetopt::TOptsParseResult parsed(&options, argc, argv);
        TSettingsType settings;

        ProcessParsedOptions(parsed, settings);
        TApp::ProcessParsedOptions(parsed, settings);

        THolder<TLog> logger = nullptr;
        try {
            logger = TLoggerFactory().MakeLogger(CreateRotatingLogBackend(
                parsed.Get("log-file"),
                parsed.Has("verbose") ? TLOG_DEBUG : TLOG_INFO,
                parsed.Has("rotate-log")));
            if (parsed.Has("json-logs")) {
                logger->SetFormatter(DeployJsonLogFormatter);
            } else {
                logger->SetFormatter(SimpleLogFormatter);
            }
        } catch (...) {
            throw yexception() << "Can't initialize logger: " << CurrentExceptionMessage();
        }

        auto& daemon(*TDaemon::Get());
        daemon.SetLoggerInstance(std::move(logger));
        daemon.SetAppInstance(MakeHolder<TApp>(*daemon.GetLogger(), settings));
        daemon.GetApp()->Setup();
        daemon.GetApp()->Run();

        return 0;
    }

} // namespace NMonitoring
