#pragma once

#include "logging.h"

#include <library/cpp/unistat/unistat.h>
#include <library/cpp/unistat/pusher/pusher.h>

#include <util/datetime/cputimer.h>

namespace NMonitoring {
    template <class TUnistatClass>
    class TMeasuredMethodBase {
    public:
        TMeasuredMethodBase(TRequestLog logger, const TString name = "", const TString yasmName = "")
            : Logger(std::move(logger))
            , Name(name)
            , YasmName(yasmName) {
            if (!Name.empty()) {
                // don't call rusage without any purpose
                UserTimer.ConstructInPlace();
            }
        }

        TMeasuredMethodBase(TLog& logger, const TString name = "", const TString yasmName = "")
            : TMeasuredMethodBase(TRequestLog(logger), name, yasmName) {
        }

        void SetRequestId(TString requestId) {
            Logger.SetRequestId(std::move(requestId));
        }

        ~TMeasuredMethodBase() {
            if (UserTimer.Defined()) {
                Logger << TLOG_INFO << "Perf: " << Name
                       << " took " << TotalTimer.Get()
                       << " (" << UserTimer->Get() << " user)";
            }
            if (!YasmName.empty()) {
                TUnistatClass::Instance().PushSignalUnsafe(YasmName, TotalTimer.Get().SecondsFloat());
            }
        }

    private:
        TRequestLog Logger;
        const TString Name;
        const TString YasmName;
        TMaybe<TProfileTimer> UserTimer;
        TSimpleTimer TotalTimer;
    };
    using TMeasuredMethod = TMeasuredMethodBase<TUnistat>;
    using TPushMeasuredMethod = TMeasuredMethodBase<TUnistatPusher>;

    class TSimpleTimedMethod {
    public:
        TSimpleTimedMethod(const std::function<void(const TString&)>& callback)
            : Callback(callback) {
        }

        ~TSimpleTimedMethod() {
            auto str = ToString(TotalTimer.Get()) + " (" + ToString(UserTimer.Get()) + " user)";
            Callback(str);
        }

    private:
        TProfileTimer UserTimer;
        TSimpleTimer TotalTimer;
        std::function<void(const TString& duration)> Callback;
    };
} // namespace NMonitoring
