#include "periodic_task.h"

#include <library/cpp/deprecated/atomic/atomic.h>

#include <util/generic/xrange.h>
#include <util/generic/ymath.h>
#include <util/system/event.h>
#include <util/system/thread.h>

namespace NMonitoring {

    class TPeriodicTask : public IPeriodicTask {
    public:
        inline TPeriodicTask(TPeriodicCallback job, const TPeriodCalculator& periodCalculator)
            : PeriodCalculator(periodCalculator)
            , Job(job)
            , Done(false)
            , Thread(new TThread(DoRun, this)) {
            Thread->Start();
        }

        static inline void* DoRun(void* data) noexcept {
            ((TPeriodicTask*)data)->Run();
            return nullptr;
        }

        inline void Run() noexcept {
            while (true) {
                Event.WaitD(PeriodCalculator.GetNextRun());

                if (AtomicGet(Done)) {
                    return;
                }

                PeriodCalculator.Increment();
                Job();
            }
        }

        ~TPeriodicTask() override {
            AtomicSet(Done, true);
            Event.Signal();
            Thread->Join();
        }

        TPeriodCalculator PeriodCalculator;
        TPeriodicCallback Job;
        TManualEvent Event;
        TAtomic Done;
        THolder<TThread> Thread;
    };

    IPeriodicTask::~IPeriodicTask() = default;

    TPeriodicTaskPtr StartPeriodicJob(TPeriodicCallback job, const TPeriodCalculator& periodCalculator) {
        return MakeAtomicShared<TPeriodicTask>(job, periodCalculator);
    }

    TPeriodCalculator MakePeriodCalculator(TInstant lastRun, const TSpecificTime& time) {
        TDuration interval;
        if (time.DayOfWeek.Defined()) {
            Y_ASSERT(time.DayOfWeek.GetRef() >= 0 && time.DayOfWeek.GetRef() <= 6);
            interval = TDuration::Days(7);
        } else if (time.Hour.Defined()) {
            Y_ASSERT(time.Hour.GetRef() >= 0 && time.Hour.GetRef() <= 23);
            interval = TDuration::Days(1);
        } else if (time.Minute.Defined()) {
            Y_ASSERT(time.Minute.GetRef() >= 0 && time.Minute.GetRef() <= 59);
            interval = TDuration::Hours(1);
        } else if (time.Second.Defined()) {
            Y_ASSERT(time.Second.GetRef() >= 0 && time.Second.GetRef() <= 59);
            interval = TDuration::Minutes(1);
        } else {
            Y_FAIL("At least one option should be defined");
        }

        // some valid time long in the past
        auto nextRun = TInstant::Zero() + TDuration::Days(time.DayOfWeek.GetOrElse(0)) + TDuration::Hours(time.Hour.GetOrElse(0)) + TDuration::Minutes(time.Minute.GetOrElse(0)) + TDuration::Seconds(time.Second.GetOrElse(0));

        nextRun += TDuration::MicroSeconds(
            ceil((TInstant::Now() - nextRun).GetValue() / (double)interval.GetValue()) * interval.GetValue());

        return TPeriodCalculator(lastRun, nextRun, interval);
    }
} // namespace NMonitoring
