#pragma once

#include <util/datetime/base.h>
#include <util/generic/function.h>
#include <util/generic/maybe.h>
#include <util/generic/ptr.h>
#include <util/generic/vector.h>
#include <util/random/random.h>

namespace NMonitoring {

    class TPeriodCalculator {
    public:
        TPeriodCalculator(TInstant lastRun, TInstant nextRun, TDuration interval, TDuration jitter = TDuration::Zero())
            : LastRun(lastRun)
            , NextRun(nextRun)
            , Interval(interval)
            , Jitter(jitter) {
        }

        TPeriodCalculator(TInstant lastRun, TDuration interval)
            : TPeriodCalculator(lastRun, Max(lastRun + interval, TInstant::Now()), interval) {
        }

        TPeriodCalculator(TInstant lastRun, TDuration interval, TDuration jitter)
            : TPeriodCalculator(lastRun, Max(lastRun + interval, TInstant::Now()), interval, jitter) {
        }

        TPeriodCalculator(TDuration interval)
            : TPeriodCalculator(TInstant::Zero(), TInstant::Now() + interval, interval) {
        }

        TPeriodCalculator(TDuration interval, bool startImmediately, TDuration jitter = TDuration::Zero())
            : TPeriodCalculator(
                TInstant::Zero(),
                startImmediately ? TInstant::Now() : TInstant::Now() + interval,
                interval,
                jitter) {
        }

        inline TInstant GetLastRun() const {
            return LastRun;
        }

        inline TInstant GetNextRun() const {
            return NextRun;
        }

        inline TDuration GetInterval() const {
            return Interval;
        }

        inline void Increment() {
            LastRun = TInstant::Now();
            NextRun = LastRun + Interval;
            if (Jitter != TDuration::Zero()) {
                NextRun += TDuration::FromValue(RandomNumber(Jitter.GetValue()));
            }
        }

    private:
        TInstant LastRun;
        TInstant NextRun;
        const TDuration Interval;
        const TDuration Jitter;
    };

    struct TSpecificTime {
        TMaybe<ui8> DayOfWeek; // [0-6], starting Monday
        TMaybe<ui8> Hour;      // [0-23], GMT time
        TMaybe<ui8> Minute;    // [0-59]
        TMaybe<ui8> Second;    // [0-59]
    };

    struct IPeriodicTask {
        virtual ~IPeriodicTask();
    };

    using TPeriodicCallback = std::function<void()>;
    using TPeriodicTaskPtr = TAtomicSharedPtr<IPeriodicTask>;

    TPeriodicTaskPtr StartPeriodicJob(TPeriodicCallback job, const TPeriodCalculator& periodCalculator);

    TPeriodCalculator MakePeriodCalculator(TInstant lastRun, const TSpecificTime& time);
} // namespace NMonitoring
