#include "periodic_task.h"

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NMonitoring;

TDuration AlmostZeroDuration = TDuration::MilliSeconds(100);

inline static void AssertAlmostSameTime(TInstant time1, TInstant time2) {
    UNIT_ASSERT(time1 - time2 <= AlmostZeroDuration);
    UNIT_ASSERT(time2 - time1 <= AlmostZeroDuration);
}

inline static void AssertTimeInRange(TInstant time, TInstant from, TInstant to) {
    UNIT_ASSERT(from - AlmostZeroDuration <= time);
    UNIT_ASSERT(time <= to + AlmostZeroDuration);
}


Y_UNIT_TEST_SUITE(TestPeriodCalculator) {
    Y_UNIT_TEST(TestSimpleInterval) {
        TPeriodCalculator calc(TInstant::Zero(), TDuration::Seconds(10));

        UNIT_ASSERT_EQUAL(calc.GetLastRun(), TInstant::Zero());
        AssertAlmostSameTime(calc.GetNextRun(), TInstant::Now());

        calc.Increment();
        AssertAlmostSameTime(calc.GetLastRun(), TInstant::Now());
        AssertAlmostSameTime(calc.GetNextRun(), TInstant::Now() + TDuration::Seconds(10));
    }

    Y_UNIT_TEST(TestIntervalWithJitter) {
        TVector<std::pair<TDuration, TDuration>> testCases = {
            {TDuration::Seconds(0), TDuration::Seconds(0)},
            {TDuration::Seconds(10), TDuration::Seconds(20)},
            {TDuration::Seconds(10), TDuration::Seconds(1000)},
            {TDuration::Seconds(10), TDuration::Days(10000)},
            {TDuration::Days(10000), TDuration::Seconds(20)},
            {TDuration::Days(10000), TDuration::Days(10000)},
        };
        for (auto& [interval, jitter] : testCases) {
            TPeriodCalculator calc(TInstant::Zero(), interval, jitter);

            UNIT_ASSERT_EQUAL(calc.GetLastRun(), TInstant::Zero());
            AssertAlmostSameTime(calc.GetNextRun(), TInstant::Now());

            calc.Increment();
            auto now = TInstant::Now();
            AssertAlmostSameTime(calc.GetLastRun(), now);
            AssertTimeInRange(calc.GetNextRun(), now + interval, now + interval + jitter);
        }
    }

    Y_UNIT_TEST(TestStartImmediately) {
        TPeriodCalculator calc(TDuration::Seconds(10), false);
        UNIT_ASSERT_EQUAL(calc.GetLastRun(), TInstant::Zero());
        AssertAlmostSameTime(calc.GetNextRun(), TInstant::Now() + TDuration::Seconds(10));

        TPeriodCalculator calcImmediate(TDuration::Seconds(10), true);
        UNIT_ASSERT_EQUAL(calcImmediate.GetLastRun(), TInstant::Zero());
        AssertAlmostSameTime(calcImmediate.GetNextRun(), TInstant::Now());
    }

    Y_UNIT_TEST(TestSpecificSecondsInterval) {
        auto calc = MakePeriodCalculator(TInstant::Zero(), TSpecificTime{.Second = 10});

        UNIT_ASSERT_EQUAL(calc.GetLastRun(), TInstant::Zero());
        UNIT_ASSERT_EQUAL(calc.GetInterval(), TDuration::Minutes(1));

        // next run less than a minute away
        UNIT_ASSERT(calc.GetNextRun() - TInstant::Now() < TDuration::Minutes(1));
        // it's also on the 10th second
        tm nextRun;
        calc.GetNextRun().GmTime(&nextRun);
        UNIT_ASSERT_EQUAL(nextRun.tm_sec, 10);

        // next run about a minute away
        calc.Increment();
        UNIT_ASSERT(calc.GetNextRun() - TInstant::Now() >= TDuration::Seconds(59));
        UNIT_ASSERT(calc.GetNextRun() - TInstant::Now() <= TDuration::Seconds(61));
    }

    Y_UNIT_TEST(TestSpecificHourAndMinuteInterval) {
        auto calc = MakePeriodCalculator(TInstant::Zero(), TSpecificTime{.Hour = 10, .Minute=30});

        UNIT_ASSERT_EQUAL(calc.GetInterval(), TDuration::Hours(24));
        // next run less than 24 hours away
        UNIT_ASSERT(calc.GetNextRun() - TInstant::Now() < TDuration::Hours(24));

        tm nextRun;
        calc.GetNextRun().GmTime(&nextRun);
        UNIT_ASSERT_EQUAL(nextRun.tm_min, 30);  // it's also on the 30th minute
        UNIT_ASSERT_EQUAL(nextRun.tm_hour, 10);  // and 10th hour GMT
    }
};
