#include "proto_replier.h"
#include "perf.h"

using namespace NMonitoring;

namespace {
    static const TString EMPTY_STRING = "";
}

void TAbstractReplier::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) {
    NMonitoring::TMeasuredMethod timer(GetLogger(), GetHandleName(), GetTimingMetricName());
    auto context(RequestToContext(request, meta, GetContentType(EProtocol::JSON)));
    RawHandle(context);
    ContextToResponse(context);
    timer.SetRequestId(std::move(context.RequestId));
}

const TString& TAbstractReplier::GetTimingMetricName() const {
    return EMPTY_STRING;
}

const TString& TAbstractReplier::GetHandleName() const {
    return EMPTY_STRING;
}

TAbstractReplier::TReplyingContext TAbstractReplier::RequestToContext(
        const NMonitoring::TServiceRequest::TRef request,
        const TParsedHttpFull& meta,
        const TString& defaultContentType) {
    auto contentType = FindHeader(request->GetInput().Headers(), "Content-Type").GetOrElse(defaultContentType);
    auto acceptType = FindHeader(request->GetInput().Headers(), "Accept").GetOrElse(contentType);
    if (acceptType == TStringBuf("*/*")) {
        acceptType = defaultContentType;
    }

    TParsedHttpFull parsedFirstLine(request->GetInput().FirstLine());
    TString content;
    if (parsedFirstLine.Method == TStringBuf("POST")) {
        content = request->GetInput().ReadAll();
    }

    return TReplyingContext{
        .RequestContent=content,
        .Request=*request,
        .Meta=meta,
        .RequestProtocol=GetProtocol(contentType),
        .ResponseProtocol=GetProtocol(acceptType)
    };
}

void TAbstractReplier::ContextToResponse(TReplyingContext& context) {
    auto acceptEncoding = FindHeader(context.Request.GetInput().Headers(), "Accept-Encoding");
    auto contentEncoding = GetContentEncoding(acceptEncoding);

    context.Response.SetHttpCode(context.ResponceCode);
    context.Response.SetContentType(GetContentType(context.ResponseProtocol));
    if (contentEncoding) {
        context.Request.EnableCompression(true);
        context.Response.AddHeader("Content-Encoding", *contentEncoding);
    }
    context.Response.SetContent(context.ResponseContent);
    context.Request.Finish(context.Response);
}

TAbstractReplier::EProtocol TAbstractReplier::GetProtocol(const TString& type) {
    if (type.Contains("application/x-protobuf")) {
        return EProtocol::PROTOBUF;
    } else if (type.Contains("application/json")) {
        return EProtocol::JSON;
    } else {
        ythrow NMonitoring::TBadRequest() << "Unknown Content Type: " << type;
    }
}

const TString& TAbstractReplier::GetContentType(EProtocol protocol) {
    switch (protocol) {
        case EProtocol::JSON: {
            static const TString json("application/json");
            return json;
        }
        case EProtocol::PROTOBUF: {
            static const TString protobuf("application/x-protobuf");
            return protobuf;
        }
    }
}

TMaybe<TString> TAbstractReplier::GetContentEncoding(const TMaybe<TString>& acceptEncoding) {
    if (acceptEncoding.Empty()) {
        return Nothing();
    } else if (acceptEncoding->Contains("*") || acceptEncoding->Contains("gzip")) {
        static const TString gzip("gzip");
        return gzip;
    } else if (acceptEncoding->Contains("z-snappy")) {
        static const TString snappy("z-snappy");
        return snappy;
    } else {
        return Nothing();
    }
}

TMaybe<TString> TAbstractReplier::FindHeader(const THttpHeaders& headers, const TStringBuf& name) {
    for (const auto& header : headers) {
        if (AsciiCompareIgnoreCase(header.Name(), name) == 0) {
            return header.Value();
        }
    }
    return Nothing();
}
