#pragma once

#include "web_server.h"

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

namespace NMonitoring {
    class TAbstractReplier : public IServiceReplier {
    public:
        virtual ~TAbstractReplier() = default;

        enum class EProtocol {
            JSON,
            PROTOBUF
        };

        struct TReplyingContext {
            TString RequestContent;
            TString ResponseContent;
            TString RequestId;

            const NMonitoring::TServiceRequest& Request;
            const TParsedHttpFull& Meta;
            THttpResponse Response;

            EProtocol RequestProtocol = EProtocol::JSON;
            EProtocol ResponseProtocol = EProtocol::JSON;
            HttpCodes ResponceCode = HttpCodes::HTTP_OK;
        };

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

        virtual void RawHandle(TReplyingContext& context) = 0;

        virtual TLog& GetLogger() = 0;
        virtual const TString& GetTimingMetricName() const;
        virtual const TString& GetHandleName() const;

    private:
        static TReplyingContext RequestToContext(const NMonitoring::TServiceRequest::TRef request,
                                                 const TParsedHttpFull& meta,
                                                 const TString& defaultContentType);
        static void ContextToResponse(TReplyingContext& context);

        static EProtocol GetProtocol(const TString& type);
        static const TString& GetContentType(EProtocol protocol);
        static TMaybe<TString> GetContentEncoding(const TMaybe<TString>& acceptEncoding);
        static TMaybe<TString> FindHeader(const THttpHeaders& headers, const TStringBuf& name);
    };

    template <class TRequestMessageType, class TResponseMessageType>
    class TAbstractProtoReplier : public TAbstractReplier {
    public:
        using Arena = google::protobuf::Arena;

        struct TProtoContext {
            TProtoContext(TReplyingContext& replyingContext)
                : ReplyingContext(replyingContext)
                , Request(*Arena::CreateMessage<TRequestMessageType>(&Arena))
                , Response(*Arena::CreateMessage<TResponseMessageType>(&Arena))
            {
            }

            TReplyingContext& ReplyingContext;
            Arena Arena;

            TRequestMessageType& Request;
            TResponseMessageType& Response;
        };

        virtual bool AllowEmptyRequest() const {
            return false;
        }

        void RawHandle(TReplyingContext& replyingContext) override {
            TProtoContext protoContext(replyingContext);
            if (!AllowEmptyRequest() || !replyingContext.RequestContent.empty()) {
                switch (replyingContext.RequestProtocol) {
                    case EProtocol::JSON: {
                        try {
                            NProtobufJson::Json2Proto(replyingContext.RequestContent, protoContext.Request);
                        } catch (...) {
                            ythrow NMonitoring::TBadRequest() << "Failed to parse json message: " << CurrentExceptionMessage();
                        }
                        break;
                    }
                    case EProtocol::PROTOBUF: {
                        if (!protoContext.Request.ParseFromString(replyingContext.RequestContent)) {
                            ythrow NMonitoring::TBadRequest() << "Failed to parse protobuf message";
                        }
                        break;
                    }
                }
            }

            Handle(protoContext);

            switch (replyingContext.ResponseProtocol) {
                case EProtocol::JSON: {
                    replyingContext.ResponseContent = NProtobufJson::Proto2Json(protoContext.Response);
                    break;
                }
                case EProtocol::PROTOBUF: {
                    replyingContext.ResponseContent = protoContext.Response.SerializeAsString();
                    break;
                }
            }
        }

        virtual void Handle(TProtoContext& context) = 0;
    };
}
