#include "proto_replier.h"
#include "proto_replier_ut.h"

#include <infra/monitoring/common/test_interface.pb.h>

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NMonitoring;

namespace {
    using THeaders = TVector<std::pair<TString, TString>>;

    class TTestReplier final : public TAbstractProtoReplier<TTestRequest, TTestResponse> {
    public:
        void Handle(TProtoContext& context) override {
            TString deadline = ToString(context.Request.GetDeadline());
            context.Response.SetMessage(deadline);
        }

        TLog& GetLogger() override {
            return Logger;
        }

        const TString& GetTimingMetricName() const override {
            static const TString metricName("test_metric");
            return metricName;
        }

    private:
        TLog Logger;
    };
}

Y_UNIT_TEST_SUITE(TestProtoReplier) {
    Y_UNIT_TEST(TestProtobuf) {
        TTestRequest request;
        request.SetDeadline(42);

        TTestReplier replier;
        auto httpResponse = THttpRequestBuilder()
            .SetContent(request.SerializeAsString())
            .AddHeaders(THeaders{
                {CONTENT_TYPE, PROTOBUF_CONTENT_TYPE}
            })
            .Execute(replier);
        UNIT_ASSERT_VALUES_EQUAL(httpResponse.Code, 200);
        UNIT_ASSERT(!httpResponse.Encoded);

        TTestResponse response;
        UNIT_ASSERT(response.ParseFromString(httpResponse.Content));
        UNIT_ASSERT_VALUES_EQUAL(response.GetMessage(), "42");
    }

    Y_UNIT_TEST(TestJson) {
        TTestRequest request;
        request.SetDeadline(42);

        TTestReplier replier;
        auto httpResponse = THttpRequestBuilder()
            .SetContent(NProtobufJson::Proto2Json(request))
            .AddHeaders(THeaders{
                {CONTENT_TYPE, JSON_CONTENT_TYPE}
            })
            .Execute(replier);
        UNIT_ASSERT_VALUES_EQUAL(httpResponse.Code, 200);
        UNIT_ASSERT(!httpResponse.Encoded);

        TTestResponse response;
        NProtobufJson::Json2Proto(httpResponse.Content, response);
        UNIT_ASSERT_VALUES_EQUAL(response.GetMessage(), "42");
    }

    Y_UNIT_TEST(TestJsonAndProtobuf) {
        TTestRequest request;
        request.SetDeadline(42);

        TTestReplier replier;
        auto httpResponse = THttpRequestBuilder()
            .SetContent(request.SerializeAsString())
            .AddHeaders(THeaders{
                {CONTENT_TYPE, PROTOBUF_CONTENT_TYPE},
                {ACCEPT, JSON_CONTENT_TYPE}
            })
            .Execute(replier);
        UNIT_ASSERT_VALUES_EQUAL(httpResponse.Code, 200);
        UNIT_ASSERT(!httpResponse.Encoded);

        TTestResponse response;
        NProtobufJson::Json2Proto(httpResponse.Content, response);
        UNIT_ASSERT_VALUES_EQUAL(response.GetMessage(), "42");
    }

    Y_UNIT_TEST(TestCompression) {
        TTestRequest request;
        request.SetDeadline(42);

        TTestReplier replier;
        auto httpResponse = THttpRequestBuilder()
            .SetContent(request.SerializeAsString())
            .AddHeaders(THeaders{
                {CONTENT_TYPE, PROTOBUF_CONTENT_TYPE},
                {ACCEPT_ENCODING, SNAPPY_ENCODING}
            })
            .Execute(replier);
        UNIT_ASSERT_VALUES_EQUAL(httpResponse.Code, 200);
        UNIT_ASSERT(httpResponse.Encoded);

        TTestResponse response;
        UNIT_ASSERT(response.ParseFromString(httpResponse.Content));
        UNIT_ASSERT_VALUES_EQUAL(response.GetMessage(), "42");
    }

    Y_UNIT_TEST(TestInvalidJson) {
        THttpRequestBuilder builder;
        builder
            .SetContent("payload")
            .AddHeaders(THeaders{
                {CONTENT_TYPE, JSON_CONTENT_TYPE}
            });

        TTestReplier replier;
        UNIT_ASSERT_EXCEPTION(builder.Execute(replier), NMonitoring::TBadRequest);
    }

    Y_UNIT_TEST(TestInvalidProtobuf) {
        THttpRequestBuilder builder;
        builder
            .SetContent("payload")
            .AddHeaders(THeaders{
                {CONTENT_TYPE, PROTOBUF_CONTENT_TYPE}
            });

        TTestReplier replier;
        UNIT_ASSERT_EXCEPTION(builder.Execute(replier), NMonitoring::TBadRequest);
    }
}
