#pragma once

#include "proto_replier.h"

namespace NMonitoring {
    using THeaders = TVector<std::pair<TString, TString>>;

    static const TString ACCEPT = "Accept";
    static const TString ACCEPT_ENCODING = "Accept-Encoding";
    static const TString CONTENT_TYPE = "Content-Type";
    static const TString CONTENT_LENGTH = "Content-Length";
    static const TString PROTOBUF_CONTENT_TYPE = "application/x-protobuf";
    static const TString JSON_CONTENT_TYPE = "application/json";
    static const TString SNAPPY_ENCODING = "z-snappy";

    struct TReplierResponse {
        TString Content;
        bool Encoded = false;
        unsigned Code = 0;

        template <class T>
        void ToProto(T& response) const {
            if (!response.ParseFromString(Content)) {
                ythrow yexception() << "wrong content given";
            }
        }
    };

    class TReplierExecutor {
    public:
        TReplierExecutor(TString inputContent)
            : InputContent(inputContent)
            , InputStream(InputContent)
            , InputHttp(&InputStream)
            , OutputStream(OutputContent)
            , OutputHttp(&OutputStream, &InputHttp)
            , Params{
                .Input=InputHttp,
                .Output=OutputHttp,
            }
            , Meta(InputHttp.FirstLine())
            , Request(MakeAtomicShared<TServiceRequest>(Params, nullptr))
        {
        }

        TReplierResponse Execute(IServiceReplier& replier) {
            replier.DoReply(Request, Meta);
            OutputHttp.Finish();

            TStringInput stream(OutputContent);
            THttpInput http(&stream);
            return TReplierResponse{
                .Content = http.ReadAll(),
                .Encoded = http.ContentEncoded(),
                .Code = ParseHttpRetCode(http.FirstLine())
            };
        }

    private:
        TString InputContent;
        TStringInput InputStream;
        THttpInput InputHttp;

        TString OutputContent;
        TStringOutput OutputStream;
        THttpOutput OutputHttp;

        TRequestReplier::TReplyParams Params;
        TParsedHttpFull Meta;
        TServiceRequest::TRef Request;
    };

    class THttpRequestBuilder {
    public:
        THttpRequestBuilder()
            : OutputStream(Output)
            , OutputHttp(&OutputStream)
        {
        }

        THttpRequestBuilder& SetContent(const TString& content) {
            Content = content;
            return *this;
        }

        THttpRequestBuilder& AddHeaders(const THeaders& headers={}) {
            for (const auto& [key, value] : headers) {
                HttpHeaders.AddHeader(key, value);
            }
            return *this;
        }

        template <class T>
        THttpRequestBuilder& FromProtoRequest(const T& request) {
            SetContent(request.SerializeAsString());
            AddHeaders(THeaders{
                {CONTENT_TYPE, PROTOBUF_CONTENT_TYPE}
            });
            return *this;
        }

        TString Build() {
            if (Content.empty()) {
                OutputHttp << "GET / HTTP/1.0\r\n";
            } else {
                OutputHttp << "POST / HTTP/1.0\r\n";
                HttpHeaders.AddHeader(CONTENT_LENGTH, ToString(Content.size()));
            }
            HttpHeaders.OutTo(&OutputHttp);
            OutputHttp << "\r\n";
            OutputHttp << Content;
            OutputHttp.Finish();
            return Output;
        }

        TReplierResponse Execute(IServiceReplier& replier) {
            TReplierExecutor executor(Build());
            return executor.Execute(replier);
        }

    private:
        TString Output;
        TStringOutput OutputStream;
        THttpOutput OutputHttp;
        THttpHeaders HttpHeaders;
        TString Content;
    };
}
