#include <atomic>

#include "united_metrics.h"
#include "metrics.h"

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/containers/stack_vector/stack_vec.h>
#include <library/cpp/unistat/unistat.h>

#include <util/string/printf.h>
#include <util/string/join.h>

namespace NMonitoring::NUnitedMetrics {
    TCustomUnistatSuffixMetricExtraProperty TCustomUnistatSuffixMetricExtraProperty::Summ("summ");
    TCustomUnistatSuffixMetricExtraProperty TCustomUnistatSuffixMetricExtraProperty::Hgram("hgram");

    namespace {
        constexpr auto DefaultUnistatPriority = NUnistat::TPriority(10);
        const TString SolomonNameKey = "sensor";

        using EUnistatAggregationType = ::EAggregationType;

        TStringBuf GetValueTypeSuffix(EValueType type) {
            switch (type) {
            case EValueType::Absolute:
                return "a";
            case EValueType::Delta:
                return "d";
            }
        }

        TStringBuf GetAggregationTypeSuffix(EAggregationType type) {
            switch (type) {
            case EAggregationType::Average:
                return "a";
            case EAggregationType::Max:
                return "x";
            case EAggregationType::Min:
                return "n";
            case EAggregationType::Sum:
                return "m";
            case EAggregationType::LastValue:
                return "t";
            }
        }

        EUnistatAggregationType ConvertAggregationType(EAggregationType type) {
            switch (type) {
            case EAggregationType::Average:
                return EUnistatAggregationType::Average;
            case EAggregationType::Max:
                return EUnistatAggregationType::Max;
            case EAggregationType::Min:
                return EUnistatAggregationType::Min;
            case EAggregationType::Sum:
                return EUnistatAggregationType::Sum;
            case EAggregationType::LastValue:
                return EUnistatAggregationType::LastValue;
            }
        }

        class TUnitedMetric: public IMetric {
        public:
            TUnitedMetric(TStackVec<IMetricPtr>&& metrics)
                : Metrics(std::move(metrics))
            {
            }

            void Update(double value) override {
                for (auto&& metric : Metrics) {
                    metric->Update(value);
                }
            }
            void Reset() override {
                for (auto&& metric : Metrics) {
                    metric->Reset();
                }
            }

            bool HasValue() const override {
                for (auto&& metric : Metrics) {
                    if (metric->HasValue()) {
                        return true;
                    }
                }
                return false;
            }
            double GetValue() const override {
                for (auto&& metric : Metrics) {
                    if (metric->HasValue()) {
                        return metric->GetValue();
                    }
                }
                return 0.;
            }
        private:
            TStackVec<IMetricPtr> Metrics;
        };

        class TUnistatFloatMetric: public IMetric {
        public:
            TUnistatFloatMetric(
                TUnistat& unistatInstance,
                const TString& name, EValueType type,
                const TAggregation& aggr, EAggregationType localAggr,
                const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps
            )
                : Hole(unistatInstance.DrillFloatHole(
                    name, MakeSuffix(type, aggr, extraProps),
                    DefaultUnistatPriority, NUnistat::TStartValue(0),
                    ConvertAggregationType(localAggr)
                ))
            {
                Y_VERIFY(localAggr != EAggregationType::Average);
            }

            void Update(double value) override {
                Hole->PushSignal(value);
            }
            void Reset() override {
                Hole->ResetSignal();
            }

            // Unistat has not so good interface to get value, so no HasValue and GetValue

        private:
            static TString MakeSuffix(
                EValueType type, const TAggregation& aggr,
                const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps
            ) {
                for (auto&& prop : extraProps) {
                    auto customSuffixProp = dynamic_cast<const TCustomUnistatSuffixMetricExtraProperty*>(&prop.get());
                    if (customSuffixProp != nullptr) {
                        return customSuffixProp->GetSuffix();
                    }
                }
                return Join(
                    "",
                    GetValueTypeSuffix(type),
                    GetAggregationTypeSuffix(aggr.Group),
                    GetAggregationTypeSuffix(aggr.MetaGroup),
                    GetAggregationTypeSuffix(aggr.Rollup)
                );
            }

        private:
            const NUnistat::IHolePtr Hole;
        };

        class TUninstatHistogramMetric: public IMetric {
        public:
            TUninstatHistogramMetric(
                TUnistat& unistatInstance,
                const TString& name, EValueType type, const TVector<double>& intervals,
                const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps
            )
                : Hole(unistatInstance.DrillHistogramHole(
                    name, MakeSuffix(type, extraProps), DefaultUnistatPriority, intervals
                ))
            {
            }

            void Update(double value) override {
                Hole->PushSignal(value);
            }
            void Reset() override {
                Hole->ResetSignal();
            }

        private:
            static TString MakeSuffix(
                EValueType type, const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps
            ) {
                for (auto&& prop : extraProps) {
                    auto customSuffixProp = dynamic_cast<const TCustomUnistatSuffixMetricExtraProperty*>(&prop.get());
                    if (customSuffixProp != nullptr) {
                        return customSuffixProp->GetSuffix();
                    }
                }
                return Join("", GetValueTypeSuffix(type), "hhh");
            }

        private:
            const NUnistat::IHolePtr Hole;
        };

        class TSolomonNumericMetric: public IMetric {
        public:
            TSolomonNumericMetric(
                IMetricRegistry& registry,
                const TString& name, EValueType type, EAggregationType localAggr
            )
                : LocalAggr(localAggr)
            {
                switch (type) {
                case EValueType::Absolute:
                    registry.LazyGauge(
                        NMonitoring::MakeLabels({{SolomonNameKey, name}}),
                        [this]() { return GetValue(); }
                    );
                    break;
                case EValueType::Delta:
                    registry.LazyCounter(
                        NMonitoring::MakeLabels({{SolomonNameKey, name}}),
                        [this]() { return GetValue(); }
                    );
                    break;
                }
            }

            void Update(double value) override {
                Update(value, LocalAggr);
            }
            void Reset() override {
                Update(0., EAggregationType::LastValue);
            }

            bool HasValue() const override {
                return true;
            }
            double GetValue() const override {
                return Value.load();
            }
        private:

            void Update(double value, EAggregationType aggr) {
                double oldValue;
                double newValue;
                do {
                    oldValue = GetValue();
                    switch (aggr) {
                    case EAggregationType::Max:
                        newValue = Max(value, oldValue);
                        break;
                    case EAggregationType::Min:
                        newValue = Min(value, oldValue);
                        break;
                    case EAggregationType::Sum:
                        newValue = value + oldValue;
                        break;
                    case EAggregationType::LastValue:
                        newValue = value;
                        break;
                    default:
                        Y_FAIL("Invalid local aggregation type");
                    }
                } while (!Value.compare_exchange_weak(oldValue, newValue));
            }

        private:
            EAggregationType LocalAggr;
            std::atomic<double> Value = {0};
        };

        class TSolomonHistogramMetric: public IMetric {
        public:
            TSolomonHistogramMetric(
                IMetricRegistry& registry,
                const TString& name, EValueType type, const THistogramBuckets& buckets, bool useIntegerBuckets
            )
                :  UseIntegerBuckets(useIntegerBuckets), Histogram(*[&] {
                    switch (type) {
                    case EValueType::Absolute:
                        return registry.HistogramCounter(
                            NMonitoring::MakeLabels({{SolomonNameKey, name}}),
                            ExplicitHistogram(ConvertBuckets(buckets))
                        );
                    case EValueType::Delta:
                        return registry.HistogramRate(
                            NMonitoring::MakeLabels({{SolomonNameKey, name}}),
                            ExplicitHistogram(ConvertBuckets(buckets))
                        );
                    }
                }())
            {
            }

            void Update(double value) override {
                if (UseIntegerBuckets) {
                    value *= 1000;
                }
                Histogram.Record(value);
            }
            void Reset() override {
                Histogram.Reset();
            }

        private:
            TBucketBounds ConvertBuckets(const THistogramBuckets& buckets) {
                TBucketBounds retval;
                retval.reserve(buckets.size());
                auto begin = buckets.begin();
                auto end = buckets.end();
                if (begin != end && std::next(begin) != end && *begin == 0.) {
                    // Solomon has open left boundary for the first bucket
                    // So convert [0, X) to (-INF, X] instead of (-INF, 0], (0, X]
                    ++begin;
                }
                retval.insert(retval.end(), begin, end);
                Y_VERIFY(retval.size() < HISTOGRAM_MAX_BUCKETS_COUNT);

                if (UseIntegerBuckets) {
                    for (unsigned long i = 0; i < retval.size(); i++) {
                        retval[i] = trunc(retval[i] * 1000);
                    }
                }

                return retval;
            }

        private:
            bool UseIntegerBuckets;
            IHistogram& Histogram;
        };
    } // anonymous namespace

    IMetricPtr TRegistry::RegisterNumeric(
        const TString& name, const TAggregation& aggr, EAggregationType localAggr,
        const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps
    ) {
        return AddMetric(name, [&] {
            return MakeAtomicShared<TUnitedMetric>(TStackVec<IMetricPtr>{
                MakeAtomicShared<TUnistatFloatMetric>(
                    UnistatInstance, name, EValueType::Absolute, aggr, localAggr, extraProps
                ),
                MakeAtomicShared<TSolomonNumericMetric>(SolomonRegistry, name, EValueType::Absolute, localAggr)
            });
        });
    }

    IMetricPtr TRegistry::RegisterCounter(
        const TString& name,
        const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps
    ) {
        return AddMetric(name, [&] {
            return MakeAtomicShared<TUnitedMetric>(TStackVec<IMetricPtr>{
                MakeAtomicShared<TUnistatFloatMetric>(
                    UnistatInstance, name,
                    EValueType::Delta, TAggregation::Sum(),
                    EAggregationType::Sum, extraProps
                ),
                MakeAtomicShared<TSolomonNumericMetric>(
                    SolomonRegistry, name, EValueType::Delta, EAggregationType::Sum
                )
            });
        });
    }

    IMetricPtr TRegistry::RegisterHistogram(
        const TString& name, EValueType type, const THistogramBuckets& buckets,
        const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps
    ) {
        return AddMetric(name, [&] {
            return MakeAtomicShared<TUnitedMetric>(TStackVec<IMetricPtr>{
                MakeAtomicShared<TUninstatHistogramMetric>(UnistatInstance, name, type, buckets, extraProps),
                MakeAtomicShared<TSolomonHistogramMetric>(SolomonRegistry, name, type, buckets, UseIntegerSolomonBuckets)
            });
        });
    }

    IMetricPtr TRegistry::AddMetric(const TString& name, const std::function<IMetricPtr ()>& constructor) {
        {
            TLightReadGuard guard(Lock);
            auto it = MetricsPerName.find(name);
            if (!it.IsEnd()) {
                return it->second;
            }
        }
        auto metric = constructor();
        TLightWriteGuard guard(Lock);
        return MetricsPerName.try_emplace(name, std::move(metric)).first->second;
    }

    IMetricPtr TRegistry::Find(const TString& name) const {
        return FindRef(name);
    }

    const IMetricPtr& TRegistry::FindRef(const TString& name) const {
        static const IMetricPtr nullMetric{};

        TLightReadGuard guard(Lock);
        auto it = MetricsPerName.find(name);
        if (!it.IsEnd()) {
            return it->second;
        }
        return nullMetric;
    }

    void TRegistry::Update(const TString& name, double value) const {
        auto& metric = FindRef(name);
        if (metric) {
            metric->Update(value);
        }
    }
    void TRegistry::Reset(const TString& name) const {
        auto& metric = FindRef(name);
        if (metric) {
            metric->Reset();
        }
    }

    TRegistry& TRegistry::Instance() {
        return *Singleton<TRegistry>(TUnistat::Instance(), *TMetricRegistry::Instance());
    }

    THistogramBuckets TRegistry::GetLinearWeights(double minValue, double maxValue, double step) {
        return TStatsInitializer::GetLinearWeights(minValue, maxValue, step);
    }

    THistogramBuckets TRegistry::GetLogWeights(double logBase, int minLog, int maxLog) {
        return TStatsInitializer::GetLogWeights(logBase, minLog, maxLog);
    }
} // namespace NMonitoring::NUnitedMetrics
