#pragma once

#include <cmath>

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/generic/ptr.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/hash.h>
#include <util/system/env.h>

class TUnistat;

namespace NMonitoring {
    class IMetricRegistry;
}

namespace NMonitoring::NUnitedMetrics {
    enum class EAggregationType {
        Average,
        Max,
        Min,
        Sum,
        LastValue,
    };

    struct TAggregation {
        EAggregationType Group;
        EAggregationType MetaGroup;
        EAggregationType Rollup;

        static TAggregation Sum() {
            return TAggregation{EAggregationType::Sum, EAggregationType::Sum, EAggregationType::Sum};
        }
        static TAggregation Max() {
            return TAggregation{EAggregationType::Max, EAggregationType::Max, EAggregationType::Max};
        }
        static TAggregation Absolute() {
            return TAggregation{EAggregationType::Sum, EAggregationType::Max, EAggregationType::Max};
        }
        static TAggregation Ammx() {
            return TAggregation{EAggregationType::Sum, EAggregationType::Sum, EAggregationType::Max};
        }
        static TAggregation Ammn() {
            return TAggregation{EAggregationType::Sum, EAggregationType::Sum, EAggregationType::Min};
        }
        static TAggregation Amxx() {
            return TAggregation{EAggregationType::Sum, EAggregationType::Max, EAggregationType::Max};
        }
    };

    enum class EValueType {
        Absolute,
        Delta,
    };

    using THistogramBuckets = TVector<double>;


    class IMetric {
    public:
        virtual ~IMetric() = default;

        virtual void Update(double value) = 0;
        virtual void Reset() = 0;

        virtual bool HasValue() const {
            return false;
        }
        virtual double GetValue() const {
            return 0.;
        }
    };

    using IMetricPtr = TAtomicSharedPtr<IMetric>;


    class IMetricExtraProperty {
    public:
        virtual ~IMetricExtraProperty() = default;
    };

    class TCustomUnistatSuffixMetricExtraProperty: public IMetricExtraProperty {
    public:
        TCustomUnistatSuffixMetricExtraProperty(const TString& suffix)
            : Suffix(suffix)
        {}

        const TString& GetSuffix() const {
            return Suffix;
        }

        static TCustomUnistatSuffixMetricExtraProperty Summ;
        static TCustomUnistatSuffixMetricExtraProperty Hgram;

    private:
        TString Suffix;
    };


    class TRegistry {
    public:
        TRegistry(TUnistat& unistatInstance, IMetricRegistry& solomonRegistry)
            : UnistatInstance(unistatInstance)
            , SolomonRegistry(solomonRegistry)
            , UseIntegerSolomonBuckets(GetEnv("USE_INTEGER_SOLOMON_BUCKETS") == "true")
        {}

        IMetricPtr RegisterNumeric(
            const TString& name, const TAggregation& aggr,
            EAggregationType localAggr = EAggregationType::LastValue,
            const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps = {}
        );
        IMetricPtr RegisterCounter(
            const TString& name,
            const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps = {}
        );
        IMetricPtr RegisterHistogram(
            const TString& name, EValueType type, const THistogramBuckets& buckets,
            const TVector<std::reference_wrapper<IMetricExtraProperty>>& extraProps = {}
        );

        IMetricPtr Find(const TString& name) const;
        void Update(const TString& name, double value) const;
        void Reset(const TString& name) const;

        TUnistat& GetUnistatInstance() {
            return UnistatInstance;
        }
        IMetricRegistry& GetSolomonRegistry() {
            return SolomonRegistry;
        }

        static TRegistry& Instance();

        static THistogramBuckets GetLinearWeights(double minValue, double maxValue, double step);
        static THistogramBuckets GetLogWeights(double logBase, int minLog, int maxLog);

    private:
        IMetricPtr AddMetric(const TString& name, const std::function<IMetricPtr ()>& constructor);

        const IMetricPtr& FindRef(const TString& name) const;

    private:
        TUnistat& UnistatInstance;
        IMetricRegistry& SolomonRegistry;

        THashMap<TString, IMetricPtr> MetricsPerName;
        TLightRWLock Lock;
        bool UseIntegerSolomonBuckets;
    };
} // namespace NMonitoring::NUnitedMetrics
