#include "united_metrics.h"

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/consumers/collecting_consumer.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/unistat/unistat.h>

using namespace NMonitoring;
using namespace NMonitoring::NUnitedMetrics;

namespace {
    const TString SolomonNameKey = "sensor";

    template<typename Callable>
    void GetSolomonMetricValue(TMetricRegistry& solomon, const TString& name, Callable cb) {
        TCollectingConsumerImpl<TLabelsImpl<TStringBuf>> consumer;
        solomon.Accept(TInstant(), &consumer);
        for (auto&& metric : consumer.Metrics) {
            auto label = metric.Labels.Get(SolomonNameKey);
            UNIT_ASSERT(label.has_value());
            if (label.value()->Value() == name) {
                UNIT_ASSERT(metric.Values);
                UNIT_ASSERT_VALUES_EQUAL(metric.Values->Size(), 1.);
                cb((*metric.Values)[0].GetValue());
                return;
            }
        }
        UNIT_ASSERT(false);
    }
    double GetSolomonNumericValue(TMetricRegistry& solomon, const TString& name) {
        double retval = 0.;
        GetSolomonMetricValue(solomon, name, [&](auto value) { retval = value.AsDouble(EMetricValueType::DOUBLE); });
        return retval;
    }
    double GetSolomonCounterValue(TMetricRegistry& solomon, const TString& name) {
        double retval = 0.;
        GetSolomonMetricValue(solomon, name, [&](auto value) { retval = value.AsDouble(EMetricValueType::INT64); });
        return retval;
    }

    double GetUnistatMetricValue(TUnistat& unistat, const TString& name) {
        auto unistatNumeric = unistat.GetSignalValueUnsafe(name);
        return unistatNumeric && unistatNumeric->HasNumber() ? unistatNumeric->GetNumber() : 0.;
    }

    using EAggrType = NUnitedMetrics::EAggregationType;
} // anonymous namespace

Y_UNIT_TEST_SUITE(UnitedMetricsRegistry) {
    Y_UNIT_TEST(TestNumericLast) {
        TUnistat unistat;
        TMetricRegistry solomon;
        TRegistry registry(unistat, solomon);

        auto numeric = registry.RegisterNumeric("numeric", TAggregation::Sum());
        UNIT_ASSERT(numeric);
        UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), "[[\"numeric_ammm\",0]]");

        numeric->Update(42.5);
        UNIT_ASSERT_VALUES_EQUAL(numeric->GetValue(), 42.5);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "numeric"), 42.5);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonNumericValue(solomon, "numeric"), 42.5);

        registry.Update("numeric", 13);
        UNIT_ASSERT_VALUES_EQUAL(numeric->GetValue(), 13);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "numeric"), 13);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonNumericValue(solomon, "numeric"), 13);

        numeric->Reset();
        UNIT_ASSERT_VALUES_EQUAL(numeric->GetValue(), 0);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "numeric"), 0);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonNumericValue(solomon, "numeric"), 0);
    }

    Y_UNIT_TEST(TestNumericLastMax) {
        TUnistat unistat;
        TMetricRegistry solomon;
        TRegistry registry(unistat, solomon);

        auto numeric = registry.RegisterNumeric("numeric", TAggregation::Sum(), EAggrType::Max);
        UNIT_ASSERT(numeric);
        UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), "[[\"numeric_ammm\",0]]");

        numeric->Update(42.5);
        UNIT_ASSERT_VALUES_EQUAL(numeric->GetValue(), 42.5);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "numeric"), 42.5);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonNumericValue(solomon, "numeric"), 42.5);

        registry.Update("numeric", 13);
        UNIT_ASSERT_VALUES_EQUAL(numeric->GetValue(), 42.5);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "numeric"), 42.5);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonNumericValue(solomon, "numeric"), 42.5);

        numeric->Update(66);
        UNIT_ASSERT_VALUES_EQUAL(numeric->GetValue(), 66);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "numeric"), 66);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonNumericValue(solomon, "numeric"), 66);

        numeric->Reset();
        UNIT_ASSERT_VALUES_EQUAL(numeric->GetValue(), 0);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "numeric"), 0);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonNumericValue(solomon, "numeric"), 0);
    }

    Y_UNIT_TEST(TestNumericSuffix) {
        auto checkSuffix = [](const TAggregation& aggr, const TStringBuf& suffix) {
            TUnistat unistat;
            TMetricRegistry solomon;
            TRegistry registry(unistat, solomon);
            registry.RegisterNumeric("numeric", aggr);
            UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), Sprintf("[[\"numeric_%s\",0]]", suffix.data()));
        };

        checkSuffix(TAggregation::Sum(), "ammm");
        checkSuffix(TAggregation::Max(), "axxx");
        checkSuffix(TAggregation::Ammx(), "ammx");
        checkSuffix(TAggregation::Amxx(), "amxx");
        checkSuffix(TAggregation::Absolute(), "amxx");
        checkSuffix(TAggregation{EAggrType::Min, EAggrType::Average, EAggrType::LastValue}, "anat");
    }

    Y_UNIT_TEST(TestCounter) {
        TUnistat unistat;
        TMetricRegistry solomon;
        TRegistry registry(unistat, solomon);

        auto counter = registry.RegisterCounter("counter");
        UNIT_ASSERT(counter);
        UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), "[[\"counter_dmmm\",0]]");

        counter->Update(42);
        UNIT_ASSERT_VALUES_EQUAL(counter->GetValue(), 42);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "counter"), 42);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonCounterValue(solomon, "counter"), 42);

        registry.Update("counter", 13);
        UNIT_ASSERT_VALUES_EQUAL(counter->GetValue(), 55);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "counter"), 55);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonCounterValue(solomon, "counter"), 55);

        counter->Reset();
        UNIT_ASSERT_VALUES_EQUAL(counter->GetValue(), 0);
        UNIT_ASSERT_VALUES_EQUAL(GetUnistatMetricValue(unistat, "counter"), 0);
        UNIT_ASSERT_VALUES_EQUAL(GetSolomonCounterValue(solomon, "counter"), 0);
    }

    Y_UNIT_TEST(TestCounterSuffix) {
        TUnistat unistat;
        TMetricRegistry solomon;
        TRegistry registry(unistat, solomon);

        registry.RegisterCounter("counter", {TCustomUnistatSuffixMetricExtraProperty::Summ});
        UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), "[[\"counter_summ\",0]]");
    }

    Y_UNIT_TEST(TestHistogram) {
        TUnistat unistat;
        TMetricRegistry solomon;
        TRegistry registry(unistat, solomon);

        // Unistat buckets: [0, 1), [1, 2), [2, ∞)
        // Solomon buckets: (∞, 1], (1, 2], (2, ∞)

        auto hist = registry.RegisterHistogram("hist", EValueType::Delta, TRegistry::GetLinearWeights(0, 2, 1));
        hist->Update(0);
        hist->Update(0);
        hist->Update(0.5);
        hist->Update(1);
        hist->Update(1.5);
        hist->Update(3);

        UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), "[[\"hist_dhhh\",[[0,3],[1,2],[2,1]]]]");
        GetSolomonMetricValue(solomon, "hist", [&](auto value) {
            auto solomonHist = value.AsHistogram();
            UNIT_ASSERT_VALUES_EQUAL(solomonHist->Count(), 3);
            UNIT_ASSERT_VALUES_EQUAL(solomonHist->Value(0), 4);
            UNIT_ASSERT_VALUES_EQUAL(solomonHist->Value(1), 1);
            UNIT_ASSERT_VALUES_EQUAL(solomonHist->Value(2), 1);
        });

        hist->Reset();

        UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), "[[\"hist_dhhh\",[[0,0],[1,0],[2,0]]]]");
        GetSolomonMetricValue(solomon, "hist", [&](auto value) {
            auto solomonHist = value.AsHistogram();
            UNIT_ASSERT_VALUES_EQUAL(solomonHist->Count(), 3);
            UNIT_ASSERT_VALUES_EQUAL(solomonHist->Value(0), 0);
            UNIT_ASSERT_VALUES_EQUAL(solomonHist->Value(1), 0);
            UNIT_ASSERT_VALUES_EQUAL(solomonHist->Value(2), 0);
        });
    }

    Y_UNIT_TEST(TestHistogramSuffix) {
        {
            TUnistat unistat;
            TMetricRegistry solomon;
            TRegistry registry(unistat, solomon);
            registry.RegisterHistogram(
                "hist", EValueType::Delta, TRegistry::GetLinearWeights(1, 2, 1),
                {TCustomUnistatSuffixMetricExtraProperty::Hgram}
            ) ;
            UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), "[[\"hist_hgram\",[[1,0],[2,0]]]]");
        }
        {
            TUnistat unistat;
            TMetricRegistry solomon;
            TRegistry registry(unistat, solomon);
            registry.RegisterHistogram("hist", EValueType::Absolute, TRegistry::GetLinearWeights(1, 2, 1));
            UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), "[[\"hist_ahhh\",[[1,0],[2,0]]]]");
        }
        {
            TUnistat unistat;
            TMetricRegistry solomon;
            TRegistry registry(unistat, solomon);
            registry.RegisterHistogram("hist", EValueType::Delta, TRegistry::GetLinearWeights(1, 2, 1));
            UNIT_ASSERT_VALUES_EQUAL(unistat.CreateJsonDump(0), "[[\"hist_dhhh\",[[1,0],[2,0]]]]");
        }
    }
}
