#pragma once

#include "http_server_metrics.h"

#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/http/server/http.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/logger/log.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/threading/future/future.h>

#include <util/generic/hash.h>
#include <util/generic/map.h>
#include <util/system/mutex.h>
#include <util/system/thread.h>
#include <util/thread/pool.h>
#include <util/datetime/cputimer.h>

#include <contrib/libs/re2/re2/re2.h>

namespace re2 {
    class RE2;
}

namespace NMonitoring {
    // code 404 Not Found
    class TNotFoundError: public yexception {
    };

    // code 400 Bad Request
    class TBadRequest: public yexception {
    };

    // code 429 Too Many Requests
    class TTooManyRequestsError: public yexception {
    };

    // code 401 Forbidden
    class TUnauthorizedError: public yexception {
    };

    // code 403 Forbidden
    class TForbiddenError: public yexception {
    };

    // code 503 Service Unavailable
    class TServiceUnavailableError: public yexception {
    };

    class TWebServer;

    class TServiceRequest: public TNonCopyable {
    public:
        using TFuture = NThreading::TFuture<void>;
        using TPromise = NThreading::TPromise<void>;
        using TRef = TAtomicSharedPtr<TServiceRequest>;

        TServiceRequest(const TRequestReplier::TReplyParams& params, THttpCallMetrics* metrics, TString remoteAddress = "")
            : RequestParams(params)
            , RemoteAddress(std::move(remoteAddress))
            , Metrics(metrics)
            , Promise(NThreading::NewPromise<void>()) {
            if (metrics) {
                metrics->CallStarted();
            }
        }

        ~TServiceRequest() {
            // don't forget about undispatched requests
            Finish();
        }

        TFuture GetFuture() {
            return Promise.GetFuture();
        }

        THttpInput& GetInput() const {
            return RequestParams.Input;
        }

        const TString& GetRemoteAddress() const {
            return RemoteAddress;
        }

        void EnableCompression(bool enable) const {
            RequestParams.Output.EnableCompression(enable);
        }

        void Finish(const THttpResponse& response) const {
            if (!Promise.HasValue()) {
                RequestParams.Output << response;
                if (Metrics) {
                    Metrics->CallCompleted(response.HttpCode(), Timer.Get());
                }
                Promise.SetValue();
            }
        }

        void Finish() const {
            if (!Promise.HasValue()) {
                if (Metrics) {
                    Metrics->CallCompleted(HttpCodes::HTTP_OK, Timer.Get());
                }
                Promise.SetValue();
            }
        }

    private:
        const TRequestReplier::TReplyParams RequestParams;
        TString RemoteAddress;
        THttpCallMetrics* Metrics;
        TSimpleTimer Timer;
        mutable TPromise Promise;
    };

    class IServiceReplier: public TNonCopyable {
    public:
        virtual ~IServiceReplier() {
        }

        virtual void DoReply(const TServiceRequest::TRef request, const TParsedHttpFull& meta) = 0;
    };

    class IServiceRegexpReplier: public TNonCopyable {
    public:
        virtual ~IServiceRegexpReplier() {
        }

        virtual void DoReply(TServiceRequest::TRef request, const TParsedHttpFull& meta,
                             const TVector<TString>& pathGroups, const TMap<int, TString>& groupNames) = 0;
    };

    struct TPathEndpoint {
        TPathEndpoint(THttpCallMetrics* metrics, IServiceReplier* Replier)
            : Metrics(metrics)
            , Replier(Replier) {
        }

        THttpCallMetrics* Metrics;
        IServiceReplier* Replier;
    };

    struct TRegexpEndpoint {
        TRegexpEndpoint(THttpCallMetrics* metrics, THolder<re2::RE2> regexp, IServiceRegexpReplier* Replier)
            : Metrics(metrics)
            , Regexp(std::move(regexp))
            , Replier(Replier) {
        }

        THttpCallMetrics* Metrics;
        THolder<re2::RE2> Regexp;
        IServiceRegexpReplier* Replier;
    };

    class TWebReplier: public TRequestReplier {
    public:
        TWebReplier(TWebServer* server, TLog& logger);

        bool DoReply(const TRequestReplier::TReplyParams& params) override;

    private:
        TWebServer* Server;
        volatile bool Ready;
        TLog& Logger;

        // this will be destroyed after call
        inline void Finish() {
            Y_VERIFY(!Ready);
            Ready = true;
            static_cast<IObjectInQueue*>(this)->Process(nullptr);
        }
    };

    class TWebServer: public ISimpleThread, public THttpServer::ICallBack {
    public:
        TWebServer(const THttpServerOptions& options, TLog& logger);

        TWebServer(const THttpServerOptions& options, THolder<IThreadPool> pool, TLog& logger);

        TWebServer(const THttpServerOptions& options)
            : TWebServer(options, TLoggerOperator<TGlobalLog>::Log()) {
        }

        ~TWebServer();

        // implements THttpServer::ICallBack
        TClientRequest* CreateClient() override;

        void* ThreadProc() noexcept override;
        void Stop();

        void StartServing();

        void SetSlowLogDuration(TDuration duration) {
            SlowLogDuration = duration;
        }

        TDuration GetSlowLogDuration() const {
            return SlowLogDuration;
        }

        void Add(const TString& path, IServiceReplier& service);
        void AddRegexp(const TString& reString, const TString& metricPath, IServiceRegexpReplier& service);

        TPathEndpoint* GetReplier(const TParsedHttpFull& parsedHttp);
        TRegexpEndpoint* GetRegexpReplier(const TParsedHttpFull& parsedHttp, TVector<TString>& pathGroups, TMap<int, TString>& groupNames);

    private:
        TLog& Logger;
        THttpServerOptions HttpOptions;
        THttpServer::TMtpQueueRef ProcessingQueue;
        THolder<THttpServer> HttpServer;
        THashMap<TString, THolder<TPathEndpoint>> Services;
        TVector<THolder<TRegexpEndpoint>> ServicesRegexp;
        int MaxNumberOfCapturingGroups = 0;
        TMutex ThreadMutex;
        THttpServerMetrics ServerMetrics;
        TDuration SlowLogDuration;
    };
} // namespace NMonitoring
