package metrics

import (
	"context"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
)

type AggregationRule int

const (
	MaxAR AggregationRule = iota
	MinAR
	SumAR
	LastAR
	AverageAR
)

type HistogramType int

const (
	AbsoluteHT HistogramType = iota
	DeltaHT
)

type HistogramTypeWrapper interface {
	Type() HistogramType
}

func (ht HistogramType) Type() HistogramType {
	return ht
}

type StructuredAggregation struct {
	Group     AggregationRule
	MetaGroup AggregationRule
	Rollup    AggregationRule
}

type Aggregation interface {
	Aggregation() StructuredAggregation
}

func (sa StructuredAggregation) Aggregation() StructuredAggregation {
	return sa
}

type CounterExtraProperty interface{}

type internalNumeric interface {
	Update(value float64)
	GetValue() float64
}

type internalHistogram interface {
	Update(value float64)
	Reset(values []float64)
	Init(bucketValues []int64)
}

type Metric interface {
	Update(value float64)
}

type Numeric struct {
	name      string
	aggr      Aggregation
	localAggr AggregationRule

	impls []internalNumeric
}

type Counter struct {
	name  string
	props []CounterExtraProperty

	impls []internalNumeric
}

type Histogram struct {
	name        string
	typeWrapper HistogramTypeWrapper
	intervals   []float64

	impls []internalHistogram
}

type RegistryHandlerFunc func(ctx context.Context, logger log.Logger, w http.ResponseWriter, req *http.Request)

type RegistryHandler struct {
	Path string
	Func RegistryHandlerFunc
}

type Registry interface {
	RegisterNumeric(numeric *Numeric)
	RegisterCounter(counter *Counter)
	RegisterHistogram(histogram *Histogram)

	WithTags(tags map[string]string) Registry
	WithPrefix(prefix string) Registry

	Handlers() []RegistryHandler

	Serialize() ([]byte, error)
}

func NewNumeric(name string, aggr Aggregation, localAggr AggregationRule) *Numeric {
	return &Numeric{
		name:      name,
		aggr:      aggr,
		localAggr: localAggr,
	}
}

func NewCounter(name string, props ...CounterExtraProperty) *Counter {
	return &Counter{
		name:  name,
		props: props,
	}
}

func NewHistogram(name string, typeWrapper HistogramTypeWrapper, intervals []float64) *Histogram {
	return &Histogram{
		name:        name,
		typeWrapper: typeWrapper,
		intervals:   intervals,
	}
}

func (n *Numeric) Update(value float64) {
	for _, impl := range n.impls {
		impl.Update(value)
	}
}

func (n *Numeric) GetValue() float64 {
	if len(n.impls) > 0 {
		return n.impls[0].GetValue()
	}
	return 0.
}

func (c *Counter) Update(value float64) {
	for _, impl := range c.impls {
		impl.Update(value)
	}
}

func (c *Counter) GetValue() float64 {
	if len(c.impls) > 0 {
		return c.impls[0].GetValue()
	}
	return 0.
}

func (h *Histogram) Update(value float64) {
	for _, impl := range h.impls {
		impl.Update(value)
	}
}

func (h *Histogram) Reset(values []float64) {
	for _, impl := range h.impls {
		impl.Reset(values)
	}
}

func (h *Histogram) Init(bucketValues []int64) {
	for _, impl := range h.impls {
		impl.Init(bucketValues)
	}
}
