package metrics

import (
	"fmt"
	"sync"
)

type MetricsMap struct {
	metrics   map[string]Metric
	onMiss    func(key string) Metric
	keyFormat string
	registry  Registry
	lock      sync.Mutex
}

func NewMetricsMap(keyFormat string, onMiss func(key string) Metric) *MetricsMap {
	return &MetricsMap{
		metrics:   make(map[string]Metric),
		onMiss:    onMiss,
		keyFormat: keyFormat,
		registry:  DefaultRegistry(),
	}
}

func (m *MetricsMap) SetRegistry(registry Registry) {
	if m.registry == registry {
		return
	}
	m.registry = registry
	for _, metric := range m.metrics {
		tryRegisterProperty(metric, m.registry)
	}
}

func (m *MetricsMap) get(key string) Metric {
	m.lock.Lock()
	defer m.lock.Unlock()
	p := m.metrics[key]
	if p == nil {
		p = m.onMiss(key)
		m.metrics[key] = p
		tryRegisterProperty(p, m.registry)
	}
	return p
}

func (m *MetricsMap) GetFor(a ...interface{}) Metric {
	return m.get(fmt.Sprintf(m.keyFormat, a...))
}

func (m *MetricsMap) Initialize(a ...interface{}) {
	m.GetFor(a...)
}

func NewCounterMap(keyFormat string, props ...CounterExtraProperty) *MetricsMap {
	return NewMetricsMap(keyFormat, func(key string) Metric {
		return NewCounter(key, props...)
	})
}

func NewAbsoluteNumericMap(keyFormat string) *MetricsMap {
	return NewNumericMap(keyFormat, AbsoluteMax(), LastAR)
}

func NewNumericMap(keyFormat string, aggr Aggregation, localAggr AggregationRule) *MetricsMap {
	return NewMetricsMap(keyFormat, func(key string) Metric {
		return NewNumeric(key, aggr, localAggr)
	})
}

func NewHistogramMap(keyFormat string, weights []float64) *MetricsMap {
	return NewMetricsMap(keyFormat, func(key string) Metric {
		return NewHistogram(key, DeltaHT, weights)
	})
}

func NewAbsoluteHistogramMap(keyFormat string, weights []float64) *MetricsMap {
	return NewMetricsMap(keyFormat, func(key string) Metric {
		return NewHistogram(key, AbsoluteHT, weights)
	})
}
