package metrics

import (
	"math"
	"reflect"
	"time"
)

func MakeLogWeights(base float64, startPower int) []float64 {
	out := make([]float64, 50)
	out[0] = 0
	for i := 0; i < 49; i++ {
		out[i+1] = math.Pow(base, float64(i+startPower))
	}
	return out
}

func tryRegisterProperty(val interface{}, registry Registry) bool {
	if registry == nil {
		return false
	}

	switch metric := val.(type) {
	case *Numeric:
		registry.RegisterNumeric(metric)
		return true
	case *Counter:
		registry.RegisterCounter(metric)
		return true
	case *Histogram:
		registry.RegisterHistogram(metric)
		return true
	}

	return false
}

func dereference(val reflect.Value) reflect.Value {
	if val.Kind() == reflect.Ptr {
		return val.Elem()
	}
	return val
}

func RegisterAllPropertiesOn(value interface{}, registry Registry) {
	if registry == nil {
		return
	}

	td := dereference(reflect.ValueOf(value))
	for i := 0; i < td.NumField(); i++ {
		field := td.Field(i)
		if !field.CanInterface() {
			continue
		}
		if tryRegisterProperty(field.Interface(), registry) {
			continue
		}
		field = dereference(field)
		if tryRegisterProperty(field.Interface(), registry) {
			continue
		} else if field.Kind() == reflect.Map {
			for _, e := range field.MapKeys() {
				tryRegisterProperty(field.MapIndex(e).Interface(), registry)
			}
		} else if field.Kind() == reflect.Struct {
			RegisterAllPropertiesOn(field.Interface(), registry)
		}
	}
}

type MetricsBackend string

const (
	GolovanMB = MetricsBackend("golovan")
	SolomonMB = MetricsBackend("solomon")
	DefaultMB = GolovanMB
)

var defaultRegistry = NewUnitedRegistry([]Registry{
	NewUnistatRegistry(UnistatRegistryOptions{}),
	NewSolomonRegistry(),
})

func DefaultRegistry() Registry {
	return defaultRegistry
}

func Register(value interface{}) {
	tryRegisterProperty(value, defaultRegistry)
}

func RegisterAllProperties(value interface{}) {
	RegisterAllPropertiesOn(value, defaultRegistry)
}

func MeasureSecondsSince(u Metric, ts time.Time) {
	u.Update(float64(time.Since(ts)) / float64(time.Second))
}
