package metrics

import (
	"encoding/json"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type childMetrics struct {
	Map    map[string]*Counter
	Simple *Counter
}

type MetricsGroupForTests struct {
	Map      map[string]*Counter
	Simple   *Counter
	Child    childMetrics
	PrtChild *childMetrics
}

func registryKeys(t *testing.T, registry *unistatRegistry) map[string]struct{} {
	result, err := registry.registry.MarshalJSON()
	require.NoError(t, err)

	var parsed [][]interface{}
	require.NoError(t, json.Unmarshal(result, &parsed))

	keys := make(map[string]struct{})
	for _, pair := range parsed {
		keys[pair[0].(string)] = struct{}{}
	}

	return keys
}

func TestMetricsRegistration(t *testing.T) {
	metric := func(name string) *Counter {
		return NewCounter(name)
	}

	m := MetricsGroupForTests{
		Map: map[string]*Counter{
			"m1": metric("root_map"),
		},
		Simple: metric("root_simple"),
		Child: childMetrics{
			Map: map[string]*Counter{
				"m1": metric("child_map"),
			},
			Simple: metric("child_simple"),
		},
		PrtChild: &childMetrics{
			Map: map[string]*Counter{
				"m1": metric("ptr_child_map"),
			},
			Simple: metric("ptr_child_simple"),
		},
	}
	registry := NewUnistatRegistry(UnistatRegistryOptions{})
	RegisterAllPropertiesOn(&m, registry)

	keys := registryKeys(t, registry.(*unistatRegistry))

	assert.Contains(t, keys, "root_map_dmmm")
	assert.Contains(t, keys, "root_simple_dmmm")
	assert.Contains(t, keys, "child_map_dmmm")
	assert.Contains(t, keys, "child_simple_dmmm")
	assert.Contains(t, keys, "ptr_child_map_dmmm")
	assert.Contains(t, keys, "ptr_child_simple_dmmm")
}

func TestMetricsRegistration_MetricsMap(t *testing.T) {
	registry := NewUnistatRegistry(UnistatRegistryOptions{})

	m := NewMetricsMap("tst_%d", func(key string) Metric {
		return NewCounter(key)
	})
	m.SetRegistry(registry)
	m.Initialize(1)
	m.Initialize(2)

	var holder struct {
		*MetricsMap
	}
	holder.MetricsMap = m

	RegisterAllPropertiesOn(holder, registry)

	{
		keys := registryKeys(t, registry.(*unistatRegistry))
		require.Len(t, keys, 2)
		assert.Contains(t, keys, "tst_1_dmmm")
		assert.Contains(t, keys, "tst_2_dmmm")
	}

	m.Initialize(5)

	{
		keys := registryKeys(t, registry.(*unistatRegistry))
		require.Len(t, keys, 3)
		assert.Contains(t, keys, "tst_1_dmmm")
		assert.Contains(t, keys, "tst_2_dmmm")
		assert.Contains(t, keys, "tst_5_dmmm")
	}
}
