package metrics

import (
	"bytes"
	"context"
	"errors"
	"math"
	"net/http"
	"os"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
)

type solomonRegistry struct {
	registry          metrics.Registry
	useIntegerBuckets bool
}

type solomonNumericAdapter struct {
	aggrRule AggregationRule

	mu    sync.RWMutex
	value float64
}

type solomonHistogramAdapter struct {
	internal       *solomon.Histogram
	multiplyValues bool
}

func NewSolomonRegistry() Registry {
	return &solomonRegistry{
		registry:          solomon.NewRegistry(&solomon.RegistryOpts{Separator: '.'}),
		useIntegerBuckets: os.Getenv("USE_INTEGER_SOLOMON_BUCKETS") == "true",
	}
}

func (r *solomonRegistry) RegisterNumeric(numeric *Numeric) {
	adapter := &solomonNumericAdapter{aggrRule: numeric.localAggr}
	r.registry.FuncGauge(numeric.name, adapter.GetValue)
	numeric.impls = append(numeric.impls, adapter)
}

func (r *solomonRegistry) RegisterCounter(counter *Counter) {
	adapter := &solomonNumericAdapter{aggrRule: SumAR}
	r.registry.(*solomon.Registry).FuncCounter(counter.name, adapter.GetIntValue)

	counter.impls = append(counter.impls, adapter)
}

func (r *solomonRegistry) RegisterHistogram(histogram *Histogram) {
	multipleValues := r.useIntegerBuckets && histogram.typeWrapper.Type() != AbsoluteHT
	if multipleValues {
		for i, interval := range histogram.intervals {
			histogram.intervals[i] = math.Trunc(interval * 1000)
		}
	}

	internal := r.registry.Histogram(histogram.name, metrics.NewBuckets(histogram.intervals...)).(*solomon.Histogram)
	if histogram.typeWrapper.Type() == DeltaHT {
		solomon.Rated(internal)
	}
	histogram.impls = append(histogram.impls, &solomonHistogramAdapter{internal, multipleValues})
}

func (r *solomonRegistry) WithTags(tags map[string]string) Registry {
	return &solomonRegistry{
		registry: r.registry.WithTags(tags),
	}
}

func (r *solomonRegistry) WithPrefix(prefix string) Registry {
	return &solomonRegistry{
		registry: r.registry.WithPrefix(prefix),
	}
}

func (r *solomonRegistry) Serialize() ([]byte, error) {
	solomonRegistry, ok := r.registry.(*solomon.Registry)
	if !ok {
		return nil, errors.New("not implemented")
	}
	buf := &bytes.Buffer{}
	_, err := solomonRegistry.StreamJSON(context.Background(), buf)
	return buf.Bytes(), err
}

func (r *solomonRegistry) handleRequest(ctx context.Context, logger log.Logger, w http.ResponseWriter, req *http.Request) {
	registry, _ := r.registry.(*solomon.Registry)

	var err error
	if req.Header.Get("Accept") == "application/x-solomon-spack" {
		_, err = registry.StreamSpack(ctx, w, solomon.CompressionLz4)
	} else {
		_, err = registry.StreamJSON(ctx, w)
	}
	if err != nil {
		logger.Errorf("Unable to render metrics handle: %s", err)
	}
}

func (r *solomonRegistry) Handlers() []RegistryHandler {
	return []RegistryHandler{{
		Path: "metrics",
		Func: r.handleRequest,
	}}
}

func (n *solomonNumericAdapter) Update(value float64) {
	n.mu.Lock()
	defer n.mu.Unlock()

	switch n.aggrRule {
	case MaxAR:
		n.value = math.Max(n.value, value)
	case MinAR:
		n.value = math.Min(n.value, value)
	case SumAR:
		n.value += value
	case LastAR:
		n.value = value
	default:
		n.value = -1
	}
}

func (n *solomonNumericAdapter) GetValue() float64 {
	n.mu.RLock()
	defer n.mu.RUnlock()

	return n.value
}

func (n *solomonNumericAdapter) GetIntValue() int64 {
	return int64(n.GetValue())
}

func (h *solomonHistogramAdapter) Update(value float64) {
	if h.multiplyValues {
		value *= 1000
	}
	h.internal.RecordValue(value)
}

func (h *solomonHistogramAdapter) Reset(values []float64) {
	h.internal.Reset()
	for _, v := range values {
		h.internal.RecordValue(v)
	}
}

func (h *solomonHistogramAdapter) Init(bucketValues []int64) {
	h.internal.InitBucketValues(bucketValues)
}
