package metrics

import (
	"context"
	"encoding/json"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/suite"

	"a.yandex-team.ru/library/go/core/metrics/solomon"
)

type SolomonMetricsTestSuite struct {
	suite.Suite

	registry Registry
}

type solomonMetric struct {
	Labels map[string]string `json:"labels"`
	Type   string            `json:"type"`
}

func (suite *SolomonMetricsTestSuite) registryMetrics() map[string]solomonMetric {
	w := httptest.NewRecorder()
	ctx := context.Background()
	_, err := suite.registry.(*solomonRegistry).registry.(*solomon.Registry).StreamJSON(ctx, w)
	suite.Require().NoError(err)

	var parsed struct {
		Metrics []solomonMetric `json:"metrics"`
	}
	suite.Require().NoError(json.Unmarshal(w.Body.Bytes(), &parsed))

	metrics := make(map[string]solomonMetric)
	for _, metric := range parsed.Metrics {
		metrics[metric.Labels["sensor"]] = metric
	}

	return metrics
}

func (suite *SolomonMetricsTestSuite) registryMetricTypes() map[string]string {
	metrics := suite.registryMetrics()

	retval := make(map[string]string)
	for name, metric := range metrics {
		retval[name] = metric.Type
	}
	return retval
}

type solomonHistogramValue struct {
	Bounds  []float64 `json:"bounds"`
	Buckets []int64   `json:"buckets"`
	Inf     int64     `json:"inf"`
}

func (suite *SolomonMetricsTestSuite) getHistValue(name string) solomonHistogramValue {
	w := httptest.NewRecorder()
	ctx := context.Background()
	_, err := suite.registry.(*solomonRegistry).registry.(*solomon.Registry).StreamJSON(ctx, w)
	suite.Require().NoError(err)

	var parsed struct {
		Metrics []struct {
			Labels    map[string]string     `json:"labels"`
			Type      string                `json:"type"`
			Histogram solomonHistogramValue `json:"hist"`
		} `json:"metrics"`
	}
	suite.Require().NoError(json.Unmarshal(w.Body.Bytes(), &parsed))

	for _, metric := range parsed.Metrics {
		if metric.Labels["sensor"] == name {
			return metric.Histogram
		}
	}
	return solomonHistogramValue{}
}

func (suite *SolomonMetricsTestSuite) SetupTest() {
	suite.registry = NewSolomonRegistry()
}

func (suite *SolomonMetricsTestSuite) TestSolomonCounter() {
	counter := NewCounter("cnt")
	suite.registry.RegisterCounter(counter)
	counter.Update(1)
	counter.Update(5)
	counter.Update(3)
	suite.Equal(9., counter.GetValue())
}

func (suite *SolomonMetricsTestSuite) TestSolomonTypes() {
	suite.registry.RegisterNumeric(NewNumeric("numeric", AbsoluteMax(), MaxAR))
	suite.registry.RegisterCounter(NewCounter("counter"))
	suite.registry.RegisterHistogram(NewHistogram("hist", AbsoluteHT, []float64{1, 2}))
	suite.registry.RegisterHistogram(NewHistogram("hist_rate", DeltaHT, []float64{1, 2}))
	metrics := suite.registryMetricTypes()
	suite.Equal(map[string]string{
		"numeric":   "DGAUGE",
		"counter":   "COUNTER",
		"hist":      "HIST",
		"hist_rate": "HIST_RATE",
	}, metrics)
}

func (suite *SolomonMetricsTestSuite) TestSolomonNumericLocalAggr() {
	{
		numeric := NewNumeric("max", AbsoluteMax(), MaxAR)
		suite.registry.RegisterNumeric(numeric)
		numeric.Update(1)
		numeric.Update(5)
		numeric.Update(3)
		suite.Equal(5., numeric.GetValue())
	}
	{
		numeric := NewNumeric("min", AbsoluteMax(), MinAR)
		suite.registry.RegisterNumeric(numeric)
		numeric.Update(-1)
		numeric.Update(-5)
		numeric.Update(-3)
		suite.Equal(-5., numeric.GetValue())
	}
	{
		numeric := NewNumeric("last", AbsoluteMax(), LastAR)
		suite.registry.RegisterNumeric(numeric)
		numeric.Update(1)
		numeric.Update(5)
		numeric.Update(3)
		suite.Equal(3., numeric.GetValue())
	}
	{
		numeric := NewNumeric("sum", AbsoluteMax(), SumAR)
		suite.registry.RegisterNumeric(numeric)
		numeric.Update(1)
		numeric.Update(5)
		numeric.Update(3)
		suite.Equal(9., numeric.GetValue())
	}
}

func (suite *SolomonMetricsTestSuite) TestUnistatSuffixInSolomon() {
	suite.registry.RegisterCounter(NewCounter("counter_no_alias"))
	suite.registry.RegisterCounter(NewCounter("counter_summ_alias", UnistatSummAlias{}))
	suite.registry.RegisterNumeric(NewNumeric("numeric_no_alias_0", AbsoluteMax(), MaxAR))
	suite.registry.RegisterNumeric(NewNumeric("numeric_no_alias_1", Absolute(), MaxAR))
	suite.registry.RegisterNumeric(NewNumeric("numeric_no_alias_2", StructuredAggregation{
		Group:     MinAR,
		MetaGroup: LastAR,
		Rollup:    AverageAR,
	}, MaxAR))
	suite.registry.RegisterNumeric(NewNumeric("numeric_max_alias", UnistatMaxAlias{}, MaxAR))
	suite.registry.RegisterHistogram(NewHistogram("hgram_no_alias_delta", DeltaHT, []float64{1, 2}))
	suite.registry.RegisterHistogram(NewHistogram("hgram_no_alias_abs", AbsoluteHT, []float64{1, 2}))
	suite.registry.RegisterHistogram(NewHistogram("hgram_with_alias", UnistatHgramAlias{}, []float64{1, 2}))

	metrics := suite.registryMetricTypes()

	suite.Contains(metrics, "counter_no_alias")
	suite.Contains(metrics, "counter_summ_alias")
	suite.Contains(metrics, "numeric_no_alias_0")
	suite.Contains(metrics, "numeric_no_alias_1")
	suite.Contains(metrics, "numeric_no_alias_2")
	suite.Contains(metrics, "numeric_max_alias")
	suite.Contains(metrics, "hgram_no_alias_delta")
	suite.Contains(metrics, "hgram_no_alias_abs")
	suite.Contains(metrics, "hgram_with_alias")
}

func (suite *SolomonMetricsTestSuite) TestSolomonHistogram() {
	hist := NewHistogram("hist", AbsoluteHT, []float64{1, 2})
	suite.registry.RegisterHistogram(hist)

	hist.Update(0)
	hist.Update(1.5)
	hist.Update(2)
	hist.Update(5)

	suite.Equal(solomonHistogramValue{
		Bounds:  []float64{1, 2},
		Buckets: []int64{1, 2},
		Inf:     1,
	}, suite.getHistValue("hist"))

	hist.Reset([]float64{100, 0.5, 0.25, 4})

	suite.Equal(solomonHistogramValue{
		Bounds:  []float64{1, 2},
		Buckets: []int64{2, 0},
		Inf:     2,
	}, suite.getHistValue("hist"))
}

func (suite *SolomonMetricsTestSuite) TestSolomonSubregistries() {
	suite.registry.RegisterCounter(NewCounter("counter_0"))

	prefixRegistry := suite.registry.WithPrefix("prefix")
	prefixRegistry.RegisterCounter(NewCounter("counter_1"))

	tagsRegistry := suite.registry.WithTags(map[string]string{"tag0": "value0", "tag1": "value1"})
	tagsRegistry.RegisterCounter(NewCounter("counter_2"))

	tagsAndPrefixRegistry := tagsRegistry.WithPrefix("prefix")
	tagsAndPrefixRegistry.RegisterCounter(NewCounter("counter_3"))

	doubleTagsRegistry := tagsRegistry.WithTags(map[string]string{"tag0": "value00", "tag2": "value2"})
	doubleTagsRegistry.RegisterCounter(NewCounter("counter_4"))

	doublePrefixRegistry := prefixRegistry.WithPrefix("another")
	doublePrefixRegistry.RegisterCounter(NewCounter("counter_5"))

	metrics := suite.registryMetrics()

	suite.Contains(metrics, "counter_0")
	suite.Contains(metrics, "prefix.counter_1")
	if suite.Contains(metrics, "counter_2") {
		suite.Equal(map[string]string{
			"tag0": "value0", "tag1": "value1", "sensor": "counter_2",
		}, metrics["counter_2"].Labels)
	}
	if suite.Contains(metrics, "prefix.counter_3") {
		suite.Equal(map[string]string{
			"tag0": "value0", "tag1": "value1", "sensor": "prefix.counter_3",
		}, metrics["prefix.counter_3"].Labels)
	}
	if suite.Contains(metrics, "counter_4") {
		suite.Equal(map[string]string{
			"tag0": "value00", "tag1": "value1", "tag2": "value2", "sensor": "counter_4",
		}, metrics["counter_4"].Labels)
	}
	suite.Contains(metrics, "prefix.another.counter_5")
}

func TestSolomon(t *testing.T) {
	suite.Run(t, new(SolomonMetricsTestSuite))
}
