package metrics

import (
	"context"
	"fmt"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/unistat"
)

type unistatRegistry struct {
	tags            map[string]string
	prefix          string
	tagStringifier  tagStringifier
	prefixSeparator string

	registry unistat.Registry
}

type unistatAggregation struct {
	internal Aggregation
}

type unistatCounterAggregation struct {
	suffix string
}

type unistatHistogramAggregation struct {
	typeWrapper HistogramTypeWrapper
}

type unistatNumericAdapter struct {
	internal *unistat.Numeric
}

type unistatHistogram interface {
	unistat.Metric
	unistat.Updater
}

type unistatHistogramAdapter struct {
	internal unistatHistogram
}

type UnistatRegistryOptions struct {
	PrefixSeparator string
	TagStringifier  tagStringifier
}

func NewUnistatRegistry(opts UnistatRegistryOptions) Registry {
	stringifier := opts.TagStringifier
	if stringifier == nil {
		stringifier = newDefaultTagStringifier()
	}
	separator := opts.PrefixSeparator
	if separator == "" {
		separator = "."
	}
	return &unistatRegistry{
		registry:        unistat.NewRegistry(),
		tagStringifier:  stringifier,
		prefixSeparator: separator,
	}
}

func (r *unistatRegistry) RegisterNumeric(numeric *Numeric) {
	impl := &unistatNumericAdapter{unistat.NewNumeric(
		r.makeFullName(numeric.name),
		0,
		&unistatAggregation{numeric.aggr},
		convertAggregationRuleToUnistat(numeric.localAggr),
	)}
	numeric.impls = append(numeric.impls, impl)
	r.registry.Register(impl.internal)
}

func (r *unistatRegistry) RegisterCounter(counter *Counter) {
	suffix := "dmmm"
	for _, prop := range counter.props {
		if alias, ok := prop.(UnistatSuffixAlias); ok {
			suffix = alias.Suffix()
		}
	}
	impl := &unistatNumericAdapter{unistat.NewNumeric(
		r.makeFullName(counter.name),
		0,
		&unistatCounterAggregation{suffix},
		unistat.Sum,
	)}
	counter.impls = append(counter.impls, impl)
	r.registry.Register(impl.internal)
}

func (r *unistatRegistry) RegisterHistogram(histogram *Histogram) {
	var impl *unistatHistogramAdapter
	switch histogram.typeWrapper.Type() {
	case AbsoluteHT:
		impl = &unistatHistogramAdapter{newUnistatAbsoluteHistogram(
			r.makeFullName(histogram.name),
			0,
			histogram.intervals,
		)}
	case DeltaHT:
		impl = &unistatHistogramAdapter{unistat.NewHistogram(
			r.makeFullName(histogram.name),
			0,
			&unistatHistogramAggregation{histogram.typeWrapper},
			histogram.intervals,
		)}
	}
	histogram.impls = append(histogram.impls, impl)
	r.registry.Register(impl.internal)
}

func (r *unistatRegistry) WithTags(tags map[string]string) Registry {
	sumTags := make(map[string]string)
	for k, v := range r.tags {
		sumTags[k] = v
	}
	for k, v := range tags {
		sumTags[k] = v
	}
	registry := &unistatRegistry{
		tags:            sumTags,
		prefix:          r.prefix,
		registry:        r.registry,
		tagStringifier:  r.tagStringifier,
		prefixSeparator: r.prefixSeparator,
	}
	return registry
}

func (r *unistatRegistry) WithPrefix(prefix string) Registry {
	newPrefix := r.prefix
	if len(newPrefix) > 0 {
		newPrefix += r.prefixSeparator
	}
	newPrefix += prefix
	return &unistatRegistry{
		tags:            r.tags,
		prefix:          newPrefix,
		registry:        r.registry,
		tagStringifier:  r.tagStringifier,
		prefixSeparator: r.prefixSeparator,
	}
}

func (r *unistatRegistry) Serialize() ([]byte, error) {
	return r.registry.MarshalJSON()
}

func (r *unistatRegistry) makeFullName(name string) string {
	retval := r.tagStringifier.ToString(r.tags)
	if len(r.prefix) > 0 {
		retval += r.prefix + r.prefixSeparator
	}
	retval += name
	return retval
}

func (r *unistatRegistry) handleRequest(_ context.Context, logger log.Logger, w http.ResponseWriter, _ *http.Request) {
	b, err := r.registry.MarshalJSON()
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		logger.Errorf("Unable to render stat handle: %s", err)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	if _, err := w.Write(b); err != nil {
		logger.Errorf("unable to write: %s", err)
		return
	}
}

func (r *unistatRegistry) Handlers() []RegistryHandler {
	return []RegistryHandler{{
		Path: "stats",
		Func: r.handleRequest,
	}}
}

func (r *unistatRegistry) writeUnistatJSON(logger log.Logger, w http.ResponseWriter) {
}

func convertAggregationRuleToUnistat(rule AggregationRule) unistat.AggregationRule {
	switch rule {
	case MaxAR:
		return unistat.Max
	case MinAR:
		return unistat.Min
	case SumAR:
		return unistat.Sum
	case LastAR:
		return unistat.Last
	case AverageAR:
		return unistat.Average
	}
	panic(fmt.Sprintf("invalid AggregationRule %d", rule))
}

func convertHistogramTypeToUnistat(hType HistogramType) unistat.AggregationType {
	switch hType {
	case AbsoluteHT:
		return unistat.Absolute
	case DeltaHT:
		return unistat.Delta
	}
	panic(fmt.Sprintf("invalid HistogramType %d", hType))
}

func (a *unistatAggregation) Suffix() string {
	if suffixAliasAggr, ok := a.internal.(UnistatSuffixAlias); ok {
		return suffixAliasAggr.Suffix()
	}
	return unistat.StructuredAggregation{
		AggregationType: unistat.Absolute,
		Group:           convertAggregationRuleToUnistat(a.internal.Aggregation().Group),
		MetaGroup:       convertAggregationRuleToUnistat(a.internal.Aggregation().MetaGroup),
		Rollup:          convertAggregationRuleToUnistat(a.internal.Aggregation().Rollup),
	}.Suffix()
}

func (a *unistatCounterAggregation) Suffix() string {
	return a.suffix
}

func (a *unistatHistogramAggregation) Suffix() string {
	if suffixAliasAggr, ok := a.typeWrapper.(UnistatSuffixAlias); ok {
		return suffixAliasAggr.Suffix()
	}
	return unistat.StructuredAggregation{
		AggregationType: convertHistogramTypeToUnistat(a.typeWrapper.Type()),
		Group:           unistat.Hgram,
		MetaGroup:       unistat.Hgram,
		Rollup:          unistat.Hgram,
	}.Suffix()
}

func (n *unistatNumericAdapter) Update(value float64) {
	n.internal.Update(value)
}

func (n *unistatNumericAdapter) GetValue() float64 {
	return n.internal.GetValue()
}

func (h *unistatHistogramAdapter) Update(value float64) {
	h.internal.Update(value)
}

func (h *unistatHistogramAdapter) Reset(values []float64) {
	if ah, ok := h.internal.(*unistatAbsoluteHistogram); ok {
		ah.Reset(values)
	} else {
		panic("Reset called on not absolute histogram")
	}
}

func (h *unistatHistogramAdapter) Init(bucketValues []int64) {
	switch hist := h.internal.(type) {
	case *unistatAbsoluteHistogram:
		hist.Init(bucketValues)
	default:
		panic("not implemented")
	}

}

type UnistatSuffixAlias interface {
	Suffix() string
}

type UnistatSummAlias struct{}

func (UnistatSummAlias) Aggregation() StructuredAggregation {
	return StructuredAggregation{
		Group:     SumAR,
		MetaGroup: SumAR,
		Rollup:    SumAR,
	}
}

func (UnistatSummAlias) Suffix() string {
	return "summ"
}

type UnistatMaxAlias struct{}

func (UnistatMaxAlias) Aggregation() StructuredAggregation {
	return StructuredAggregation{
		Group:     MaxAR,
		MetaGroup: MaxAR,
		Rollup:    MaxAR,
	}
}

func (UnistatMaxAlias) Suffix() string {
	return "max"
}

type UnistatHgramAlias struct{}

func (UnistatHgramAlias) Type() HistogramType {
	return DeltaHT
}

func (UnistatHgramAlias) Suffix() string {
	return "hgram"
}

type tagStringifier interface {
	ToString(tags map[string]string) string
}

type defaultTagStringifier struct{}

func newDefaultTagStringifier() tagStringifier {
	return defaultTagStringifier{}
}

func (defaultTagStringifier) ToString(tags map[string]string) string {
	retval := ""
	for k, v := range tags {
		retval += k + "=" + v + ";"
	}
	return retval
}
