package metrics

import (
	"encoding/json"
	"sync"

	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

// unistatAbsoluteHistogram implements Metric + Updater interface
type unistatAbsoluteHistogram struct {
	mu       sync.RWMutex
	name     string
	priority unistat.Priority

	intervals []float64
	weights   []int64
}

func newUnistatAbsoluteHistogram(name string, priority unistat.Priority, intervals []float64) *unistatAbsoluteHistogram {
	return &unistatAbsoluteHistogram{
		name:      name,
		priority:  priority,
		intervals: intervals,
		weights:   make([]int64, len(intervals)),
	}
}

func (h *unistatAbsoluteHistogram) Name() string {
	return h.name
}

func (h *unistatAbsoluteHistogram) Priority() unistat.Priority {
	return h.priority
}

func (h *unistatAbsoluteHistogram) Aggregation() unistat.Aggregation {
	return aggr.AbsoluteHistogram()
}

func (h *unistatAbsoluteHistogram) Update(value float64) {
	h.mu.Lock()
	defer h.mu.Unlock()

	for i := len(h.intervals); i > 0; i-- {
		if value >= h.intervals[i-1] {
			h.weights[i-1]++
			break
		}
	}
}

func (h *unistatAbsoluteHistogram) Reset(values []float64) {
	h.mu.Lock()
	defer h.mu.Unlock()
	// this is not super efficient but doing it faster requires sorting the values array or doing some kind of allocations
	for i := 0; i < len(h.weights); i++ {
		h.weights[i] = 0
	}
	for _, value := range values {
		for i := len(h.intervals); i > 0; i-- {
			if value >= h.intervals[i-1] {
				h.weights[i-1]++
				break
			}
		}
	}
}

func (h *unistatAbsoluteHistogram) Init(bucketValues []int64) {
	h.mu.RLock()
	defer h.mu.RUnlock()

	h.weights = make([]int64, len(h.weights))
	copy(h.weights, bucketValues)
}

func (h *unistatAbsoluteHistogram) MarshalJSON() ([]byte, error) {
	h.mu.RLock()
	defer h.mu.RUnlock()

	buckets := [][2]interface{}{}
	for i := range h.intervals {
		b := h.intervals[i]
		w := h.weights[i]
		buckets = append(buckets, [2]interface{}{b, w})
	}

	jsonName := h.name + "_" + h.Aggregation().Suffix()
	return json.Marshal([]interface{}{jsonName, buckets})
}
