package metrics

import (
	"encoding/json"
	"strings"
	"testing"

	"github.com/stretchr/testify/suite"
)

type UnistatMetricsTestSuite struct {
	suite.Suite

	registry Registry
}

func (suite *UnistatMetricsTestSuite) parseName(fullname string) (string, map[string]string) {
	parts := strings.Split(fullname, ";")
	name := ""
	tags := make(map[string]string)
	for _, part := range parts {
		kv := strings.SplitN(part, "=", 2)
		if len(kv) == 1 {
			name = part
		} else if len(kv) == 2 {
			tags[kv[0]] = kv[1]
		}
	}
	return name, tags
}

func (suite *UnistatMetricsTestSuite) registryKeys() map[string]map[string]string {
	result, err := suite.registry.(*unistatRegistry).registry.MarshalJSON()
	suite.Require().NoError(err)

	var parsed [][]interface{}
	suite.Require().NoError(json.Unmarshal(result, &parsed))

	keys := make(map[string]map[string]string)
	for _, pair := range parsed {
		name, tags := suite.parseName(pair[0].(string))
		keys[name] = tags
	}

	return keys
}

func (suite *UnistatMetricsTestSuite) SetupTest() {
	suite.registry = NewUnistatRegistry(UnistatRegistryOptions{})
}

func (suite *UnistatMetricsTestSuite) TestUnistatSuffix() {
	suite.registry.RegisterCounter(NewCounter("counter_no_alias"))
	suite.registry.RegisterCounter(NewCounter("counter_summ_alias", UnistatSummAlias{}))
	suite.registry.RegisterNumeric(NewNumeric("numeric_no_alias_0", AbsoluteMax(), MaxAR))
	suite.registry.RegisterNumeric(NewNumeric("numeric_no_alias_1", Absolute(), MaxAR))
	suite.registry.RegisterNumeric(NewNumeric("numeric_no_alias_2", StructuredAggregation{
		Group:     MinAR,
		MetaGroup: LastAR,
		Rollup:    AverageAR,
	}, MaxAR))
	suite.registry.RegisterNumeric(NewNumeric("numeric_max_alias", UnistatMaxAlias{}, MaxAR))
	suite.registry.RegisterHistogram(NewHistogram("hgram_no_alias_delta", DeltaHT, []float64{1, 2}))
	suite.registry.RegisterHistogram(NewHistogram("hgram_no_alias_abs", AbsoluteHT, []float64{1, 2}))
	suite.registry.RegisterHistogram(NewHistogram("hgram_with_alias", UnistatHgramAlias{}, []float64{1, 2}))

	keys := suite.registryKeys()

	suite.Contains(keys, "counter_no_alias_dmmm")
	suite.Contains(keys, "counter_summ_alias_summ")
	suite.Contains(keys, "numeric_no_alias_0_axxx")
	suite.Contains(keys, "numeric_no_alias_1_ammm")
	suite.Contains(keys, "numeric_no_alias_2_antv")
	suite.Contains(keys, "numeric_max_alias_max")
	suite.Contains(keys, "hgram_no_alias_delta_dhhh")
	suite.Contains(keys, "hgram_no_alias_abs_ahhh")
	suite.Contains(keys, "hgram_with_alias_hgram")
}

func (suite *UnistatMetricsTestSuite) TestUnistatSubregistries() {
	suite.registry.RegisterCounter(NewCounter("counter_0"))

	prefixRegistry := suite.registry.WithPrefix("prefix")
	prefixRegistry.RegisterCounter(NewCounter("counter_1"))

	tagsRegistry := suite.registry.WithTags(map[string]string{"tag0": "value0", "tag1": "value1"})
	tagsRegistry.RegisterCounter(NewCounter("counter_2"))

	tagsAndPrefixRegistry := tagsRegistry.WithPrefix("prefix")
	tagsAndPrefixRegistry.RegisterCounter(NewCounter("counter_3"))

	doubleTagsRegistry := tagsRegistry.WithTags(map[string]string{"tag0": "value00", "tag2": "value2"})
	doubleTagsRegistry.RegisterCounter(NewCounter("counter_4"))

	doublePrefixRegistry := prefixRegistry.WithPrefix("another")
	doublePrefixRegistry.RegisterCounter(NewCounter("counter_5"))

	keys := suite.registryKeys()

	suite.Contains(keys, "counter_0_dmmm")
	suite.Contains(keys, "prefix.counter_1_dmmm")
	if suite.Contains(keys, "counter_2_dmmm") {
		suite.Equal(map[string]string{"tag0": "value0", "tag1": "value1"}, keys["counter_2_dmmm"])
	}
	if suite.Contains(keys, "prefix.counter_3_dmmm") {
		suite.Equal(map[string]string{"tag0": "value0", "tag1": "value1"}, keys["prefix.counter_3_dmmm"])
	}
	if suite.Contains(keys, "counter_4_dmmm") {
		suite.Equal(map[string]string{"tag0": "value00", "tag1": "value1", "tag2": "value2"}, keys["counter_4_dmmm"])
	}
	suite.Contains(keys, "prefix.another.counter_5_dmmm")
}

func (suite *UnistatMetricsTestSuite) TestUnistatRegistryOptions() {
	const (
		tagsToString = "_"
		separator    = "_sep_"
		metricName   = "metric_name"
		prefix       = "pref"
		tagKey       = "k"
		tagVal       = "v"
	)
	options := []UnistatRegistryOptions{
		{},
		{
			TagStringifier:  newMockStringifier(tagsToString),
			PrefixSeparator: separator,
		},
	}
	expectedNames := []string{
		tagKey + "=" + tagVal + ";" + prefix + "." + metricName,
		tagsToString + prefix + separator + metricName,
	}
	registries := make([]Registry, len(options))
	for i, opts := range options {
		registries[i] = NewUnistatRegistry(opts)
	}
	for i, r := range registries {
		r := r.WithPrefix(prefix).WithTags(map[string]string{tagKey: tagVal})
		fullName := r.(*unistatRegistry).makeFullName(metricName)
		suite.Equal(expectedNames[i], fullName)
	}
}

func TestUnistat(t *testing.T) {
	suite.Run(t, new(UnistatMetricsTestSuite))
}

type mockTagStringifier struct {
	value string
}

func newMockStringifier(value string) mockTagStringifier {
	return mockTagStringifier{value: value}
}

func (m mockTagStringifier) ToString(map[string]string) string {
	return m.value
}
