package client

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net"
	"net/http"
	"time"
)

type InstanceType int32

const (
	InstanceTypeUnknown              InstanceType = 0
	InstanceTypeInstanceList         InstanceType = 1
	InstanceTypeExtendedGencfgGroups InstanceType = 5
	InstanceTypeYPPodIDs             InstanceType = 9
	InstanceTypeAllocationPods       InstanceType = 10
)

var InstanceTypeMapping = map[string]InstanceType{
	"INSTANCE_LIST":          InstanceTypeInstanceList,
	"EXTENDED_GENCFG_GROUPS": InstanceTypeExtendedGencfgGroups,
	"YP_POD_IDS":             InstanceTypeYPPodIDs,
	"ALLOCATION_PODS":        InstanceTypeAllocationPods,
}

type StaticFile struct {
	LocalPath     string `json:"local_path"`
	IsDynamic     bool   `json:"is_dynamic"`
	DownloadQueue string `json:"download_queue"`
	DownloadSpeed int    `json:"download_speed"`
	Storage       string `json:"storage"`
	Content       string `json:"content"`
	ExtractPath   string `json:"extract_path"`
	CheckPeriod   string `json:"check_period"`
}

type SandboxFile struct {
	LocalPath     string `json:"local_path"`
	IsDynamic     bool   `json:"is_dynamic"`
	TaskID        string `json:"task_id"`
	TaskType      string `json:"task_type"`
	ResourceType  string `json:"resource_type"`
	DownloadQueue string `json:"download_queue"`
	DownloadSpeed int    `json:"download_speed"`
	Storage       string `json:"storage"`
	ResourceID    string `json:"resource_id"`
	ExtractPath   string `json:"extract_path"`
	CheckPeriod   string `json:"check_period"`
}

type URLFile struct {
	LocalPath     string `json:"local_path"`
	IsDynamic     bool   `json:"is_dynamic"`
	URL           string `json:"url"`
	Checksum      string `json:"chksum"`
	DownloadQueue string `json:"download_queue"`
	DownloadSpeed int    `json:"download_speed"`
	Storage       string `json:"storage"`
	ExtractPath   string `json:"extract_path"`
	CheckPeriod   string `json:"check_period"`
}

type ISSHookTimeLimits struct {
	MinRestartPeriod     *uint64 `json:"min_restart_period" yaml:"min_restart_period"`
	MaxExecutionTime     *uint64 `json:"max_execution_time" yaml:"max_execution_time"`
	RestartPeriodBackoff *uint64 `json:"restart_period_backoff" yaml:"restart_period_backoff"`
	MaxRestartPeriod     *uint64 `json:"max_restart_period" yaml:"max_restart_period"`
	RestartPeriodScale   *uint64 `json:"restart_period_scale" yaml:"restart_period_scale"`
}

type ISS3HooksTimeLimits struct {
	ISSHookStart      *ISSHookTimeLimits `json:"iss_hook_start"`
	ISSHookStop       *ISSHookTimeLimits `json:"iss_hook_stop"`
	ISSHookStatus     *ISSHookTimeLimits `json:"iss_hook_status"`
	ISSHookInstall    *ISSHookTimeLimits `json:"iss_hook_install"`
	ISSHookUninstall  *ISSHookTimeLimits `json:"iss_hook_uninstall"`
	ISSHookValidate   *ISSHookTimeLimits `json:"iss_hook_validate"`
	ISSHookNotify     *ISSHookTimeLimits `json:"iss_hook_notify"`
	ISSHookReopenlogs *ISSHookTimeLimits `json:"iss_hook_reopenlogs"`
}

func (l *ISS3HooksTimeLimits) GetTimeLimitsByName(name string) (*ISSHookTimeLimits, error) {
	if name == "iss_hook_start" {
		return l.ISSHookStart, nil
	} else if name == "iss_hook_stop" {
		return l.ISSHookStop, nil
	} else if name == "iss_hook_status" {
		return l.ISSHookStatus, nil
	} else if name == "iss_hook_install" {
		return l.ISSHookInstall, nil
	} else if name == "iss_hook_uninstall" {
		return l.ISSHookUninstall, nil
	} else if name == "iss_hook_validate" {
		return l.ISSHookValidate, nil
	} else if name == "iss_hook_notify" {
		return l.ISSHookNotify, nil
	} else if name == "iss_hook_reopenlogs" {
		return l.ISSHookReopenlogs, nil
	}
	return nil, fmt.Errorf("unknown iss hook name: %s", name)
}

type ISS3HooksResourceLimits struct {
	ISSHookStartCPUWeight *uint64 `json:"iss_hook_start_cpu_weight"`
}

type ISSSettings struct {
	InstanceCls        string                  `json:"instance_cls"`
	HookTimeLimits     ISS3HooksTimeLimits     `json:"hooks_time_limits"`
	HookResourceLimits ISS3HooksResourceLimits `json:"hooks_resource_limits"`
}

func (s *ISSSettings) UnmarshalJSON(text []byte) error {
	// issSettingsAlias is needed to prevent recursive calls to UnmarshalJSON.
	// This works because a new type has no methods defined.
	type issSettingsAlias ISSSettings
	// set default value for InstanceCls
	settings := issSettingsAlias{
		InstanceCls: "ru.yandex.iss.Instance",
	}
	if err := json.Unmarshal(text, &settings); err != nil {
		return err
	}
	*s = ISSSettings(settings)
	return nil
}

type OrthogonalTags struct {
	IType   string `json:"itype"`
	CType   string `json:"ctype"`
	Prj     string `json:"prj"`
	MetaPrj string `json:"metaprj"`
	Tier    string `json:"tier"`
}

type SysctlParam struct {
	Name  string `json:"name"`
	Value string `json:"value"`
}

type SysctlSettings struct {
	Params []SysctlParam `json:"params"`
}

func (s *SysctlSettings) ToPortoString() (string, error) {
	b := &bytes.Buffer{}
	for _, p := range s.Params {
		if _, err := fmt.Fprintf(b, "%s:%s", p.Name, p.Value); err != nil {
			return "", err
		}
	}
	return b.String(), nil
}

type YPPod struct {
	Cluster string `json:"cluster"`
	PodID   string `json:"pod_id"`
}

type YPPodIDs struct {
	Pods           []YPPod        `json:"pods"`
	OrthogonalTags OrthogonalTags `json:"orthogonal_tags"`
	SysctlSettings SysctlSettings `json:"sysctl_settings"`
}

type Instances struct {
	ChosenType  string      `json:"chosen_type"`
	YPPodIDs    YPPodIDs    `json:"yp_pod_ids"`
	ISSSettings ISSSettings `json:"iss_settings"`
}

func (i *Instances) ChosenTypeMapped() (InstanceType, error) {
	chType, ok := InstanceTypeMapping[i.ChosenType]
	if !ok {
		return InstanceTypeUnknown, fmt.Errorf("unknown instance type: %s", i.ChosenType)
	}
	return chType, nil
}

type Resources struct {
	StaticFiles  []StaticFile  `json:"static_files"`
	SandboxFiles []SandboxFile `json:"sandbox_files"`
	URLFiles     []URLFile     `json:"url_files"`
}

type Engines struct {
	EngineType string `json:"engine_type"`
}

type RuntimeAttrsContent struct {
	Resources Resources `json:"resources"`
	Instances Instances `json:"instances"`
	Engines   Engines   `json:"engines"`
}

type RuntimeAttrs struct {
	ServiceID  string              `json:"service_id"`
	SnapshotID string              `json:"_id"`
	Content    RuntimeAttrsContent `json:"content"`
}

type NannyClient struct {
	apiURL     string
	timeout    time.Duration
	token      string
	httpClient http.Client
}

func NewNannyClient(apiURL string, token string, timeout time.Duration, connectionTimeout time.Duration) *NannyClient {
	httpClient := http.Client{}
	httpClient.Transport = &http.Transport{
		Dial: (&net.Dialer{Timeout: time.Duration(connectionTimeout) * time.Millisecond}).Dial,
	}
	return &NannyClient{
		apiURL:     apiURL,
		token:      token,
		timeout:    time.Duration(timeout) * time.Second,
		httpClient: httpClient,
	}
}

func (c *NannyClient) GetRuntimeAttributeSnapshot(ctx context.Context, serviceID string, snapshotID string) (*RuntimeAttrs, error) {
	url := fmt.Sprintf("%s/services/%s/history/runtime_attrs/%s/", c.apiURL, serviceID, snapshotID)
	ctx, cancel := context.WithTimeout(ctx, c.timeout)
	defer cancel()

	req, err := http.NewRequestWithContext(ctx, "GET", url, nil)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", c.token))

	res, err := http.DefaultClient.Do(req)
	if err != nil {
		return nil, err
	}
	body, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return nil, err
	}

	ra := &RuntimeAttrs{}
	if err = json.Unmarshal(body, ra); err != nil {
		return nil, fmt.Errorf("failed to unmarshal JSON response from Nanny: %s", err.Error())
	}
	return ra, nil
}
