"""
Библиотека для получения инстансов сервися из Nanny/HQ: https://wiki.yandex-team.ru/jandekspoisk/sepe/nanny/hq/
Над библиотекой есть простой консольный клиент. Лежит в arcadia/infra/nanny/hq_get_instances/app/

Пример 1: получить имена всех активных инстансов сервиса.

import infra.nanny.hq_get_instances.lib as hq

all_antirobots = hq.extract_node_names_sorted(
    instances=hq.filter_instances_just_active(
        hq.get_instances('production_antirobot_iss', retries=3, request_timeout=5)
    )
)


Пример 2: получить имена активных инстансов сервиса в локации.

import infra.nanny.hq_get_instances.lib as hq

remote_wizards = hq.extract_node_names_sorted(
    hq.filter_instances_by_tag(tag='a_geo_sas',
        instances=hq.filter_instances_just_active(
            hq.get_instances('production_wizard_antirobot', retries=3, request_timeout=5)
        )
    )
)


Пример 3: получить имена инстансов сервиса для "текущей ревизии".
  Может использоваться при старте сервиса, чтобы узнать свою топологию, даже если не все инстансы уже подняты.
  Note: свою ревизию инстанс может узнать из `grep configurationId dump.json` - https://wiki.yandex-team.ru/iss3/specifications/agent/usecases/#usecase10.polucheniekollekciientity

import infra.nanny.hq_get_instances.lib as hq

hq.extract_node_names_sorted(
    hq.filter_instances_by_revision(revision='production_antirobot_iss_prestable-1491923304116',
        instances=hq.get_instances('production_antirobot_iss_prestable', retries=3, request_timeout=5)
    )
)

"""
from nanny_rpc_client import RequestsRpcClient

# Определения запросов и ответов сервера
from clusterpb import federated_stub, federated_pb2
from clusterpb import hq_pb2, hq_stub


import requests.exceptions


def get_instances(service_name, retries=3, request_timeout=5):
    """
    Получаем инстансы Nanny-сервиса. Без какой либо фильтрации.

    :param service_name: Имя Nanny-сервиса; например: balancer_l7_fast_test; список сервисов: https://nanny.yandex-team.ru/ui/#/services/
    :param retries: Число попыток. Перехватываются только ConnectionError.
    :param request_timeout: Таймаут _одной_ попытки в секундах.
    :returns: list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    """
    for attempt in xrange(retries + 1):
        try:
            return _get_instances(service_name, request_timeout)
        except:
            if attempt == retries:
                raise


def _timed(function, *args, **kwargs):
    """
    Возвращаем результат выполнения функции и время ее выполнения.

    :param function: Целевая функция.
    :returns: (result, elapsed_time) - кортеж из результата выполнения функции и времени выполнения.
    """
    from time import time
    start = time()
    result = function(*args, **kwargs)
    end = time()
    return result, end - start


_federated_api_url = 'http://federated.yandex-team.ru/rpc/federated/'
_instances_url_suffix = 'rpc/instances/'


def _get_instances(service_name, request_timeout):
    """
    Пример с https://wiki.yandex-team.ru/JandeksPoisk/Sepe/nanny/hq/api/
    В этом примере мы:
     * используем federated сервис для получения всех кластеров HQ
     * затем у каждого запросим инстансы интересующего нас сервиса
    """
    client = RequestsRpcClient(_federated_api_url, request_timeout=request_timeout)
    # Инициализируем клиент для Federated Service'а
    f_client = federated_stub.FederatedClusterServiceStub(client)
    # Создаём объект запроса, без параметров
    find_req = federated_pb2.FindClustersRequest()
    # Выполняем запрос и сохраняем список кластеров
    clusters_resp, elapsed_time = _timed(f_client.find_clusters, find_req)
    clusters = clusters_resp.value
    request_timeout -= elapsed_time
    instances = []  # Здесь будем хранить список инстансов сервиса
    # Создаём объект запроса (он будет один для всех кластеров)
    find_req = hq_pb2.FindInstancesRequest()
    # Указываем интересующий нас сервис
    find_req.filter.service_id = service_name
    # Выполняем поиск инстансов в каждом кластере
    for c in clusters:
        # Создаём клиент для конкретного кластера
        endpoint_url = c.spec.endpoint.url + _instances_url_suffix
        if request_timeout < 0.0:
            raise requests.exceptions.Timeout
        client = RequestsRpcClient(endpoint_url, request_timeout=request_timeout)
        hq_client = hq_stub.InstanceServiceStub(client)
        resp, elapsed_time = _timed(hq_client.find_instances, find_req)
        request_timeout -= elapsed_time
        instances.extend(resp.instance)
    return instances


def filter_instances_just_active_revisions(instances):
    """
    Оставляем _все_ инстансы, которые входят в ревизии, в которых есть хотя бы один живой инстанс.
    Интанс не обязан быть живым.

    :param instances: list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    :returns:         list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    """
    instances_list = []
    instances_list.extend(instances)
    from collections import defaultdict
    instances_by_revision = defaultdict(list)
    active_revision_ids = set()
    for instance_index, instance in enumerate(instances_list):
        for revision in instance.status.revision:
            instances_by_revision[revision.id].append(instance_index)
            if revision.ready.status == 'True':
                active_revision_ids.add(revision.id)
    import itertools
    unique_active_instance_indexes = sorted(set(itertools.chain.from_iterable(
        instances_by_revision[active_revision_id] for active_revision_id in active_revision_ids)))
    return map(lambda instance_index: instances_list[instance_index], unique_active_instance_indexes)


def filter_instances_just_active(instances):
    """
    Оставляем инстансы, в которых есть хотя бы одна _активная_ ревизия.
    Скорее всего, вам лучше подойдет filter_instances_just_active_revisions().
    Эта функция (filter_instances_just_active) не вернет инстансы с дохлых в данный момент тачек.

    :param instances: list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    :returns:         list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    """
    return filter(lambda instance: instance.status.ready.status == 'True',  instances)


def filter_instances_by_tag(instances, tag):
    """
    Оставляем инстансы, в которых есть хотя бы одна ревизия, которая имела заданный тег.

    :param instances: list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    :params tag:      Пример: 'a_geo_msk'
    :returns:         list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    """
    return filter(lambda instance: any(any(tag == tag_ for tag_ in revision.tags) for revision in instance.spec.revision),  instances)


def filter_instances_by_revision(instances, revision):
    """
    Оставляем инстансы, в которых есть интересующая ревизия. Ревизия не обязаны быть активной.

    :param instances: list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    :param revision:  Пример: production_antirobot_iss_prestable-1491923304116
    :returns:         list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    """
    return filter(lambda instance: any(revision_.id == revision for revision_ in instance.spec.revision),  instances)


def extract_node_names(instances):
    """
    Оставляет от каждого clusterpb.types_pb2.Instance только node_name.

    :param instances: list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    :returns:         list of string. Например: ['ws14-011.search.yandex.net', 'sas1-2218.seach.yandex.net',  'man1-3249.seach.yandex.net']
    """
    return map(lambda instance: instance.spec.node_name,  instances)


def extract_node_names_sorted(instances):
    """
    Аналогично extract_node_names(), только уникализирует и сортрует имена.

    :param instances: list объектов описывающих инстансы типа clusterpb.types_pb2.Instance
    :returns:         list of string. Например: ['man1-3249.seach.yandex.net', 'sas1-2218.seach.yandex.net', 'ws14-011.search.yandex.net']
    """
    return sorted(set(extract_node_names(instances)))
