#!/usr/bin/env python3
"""
First and fast implementation of protobuf compiler plugin which generates
HTTP RPC clients for Python.
Implementation assumes that given client has following interface:

class IClient(object):
    def call_remote_method(method_name, request_protobuf, response_protobuf, **kwargs):
        raise NotImplemented

Method does not return anything.
"""
import io
import sys

from google.protobuf.compiler import plugin_pb2 as plugin

INTRO = """# THIS FILE IS AUTOMATICALLY GENERATED
# PLEASE DO NOT EDIT
"""

INIT_COMMENT = '""":param client: Http Rpc Client implementation"""'
SERVICE_INIT = ("def __init__(self, client):\n"
                "    {}\n"
                "    self.client = client\n".format(INIT_COMMENT))

METHOD_BODY = """
response = {response_class}()
self.client.call_remote_method('{method_name}',
                               {request_protobuf},
                               response, **kwargs)
return response
""".lstrip('\n')


class Indenter(object):
    def __init__(self, output, indent=4):
        self.output = output
        self.buf = io.StringIO()
        self.indent = ' ' * indent

    def write(self, b):
        self.buf.write(b)

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.buf.seek(0)
        for line in self.buf:
            self.output.write(self.indent)
            self.output.write(line)


def convert_to_snake_case(method_name):
    b = io.StringIO()
    first_time = True
    for c in method_name:
        if c.isupper():
            if first_time:
                first_time = False
            else:
                b.write('_')
            b.write(c.lower())
        else:
            b.write(c)
    return b.getvalue()


def strip_fully_qualified_name(full_method_name):
    """
    Transforms .nanny.repo.ListSummariesRequest into ListSummariesRequest.
    """
    return full_method_name.rpartition('.')[-1]


def write_method_body(buf, method):
    # output_type looks like '.nanny.queues.ListQueuesResponse'
    # split it into ['', 'queues', 'ListQueuesResponse']
    qualifiers = method.output_type.split('.')
    # We assume two things here:
    #  * All request/response are defined in same file as a service
    #  * File with service is processed by native protobuf compiler and outputs
    #  files name like 'package_name_pb2.py' - so that imports work
    response_class = '{}_pb2.{}'.format(qualifiers[-2], qualifiers[-1])
    input_type = convert_to_snake_case(strip_fully_qualified_name(method.input_type))
    buf.write(METHOD_BODY.format(
        response_class=response_class,
        method_name=method.name,
        request_protobuf=input_type,
    ))


def generate_service(service_descriptor):
    buf = io.StringIO()
    buf.write("class {}Stub(object):\n".format(service_descriptor.name))
    with Indenter(buf) as indent:
        indent.write(SERVICE_INIT)
    for method in service_descriptor.method:
        buf.write('\n')
        method_name = convert_to_snake_case(method.name)
        input_type = convert_to_snake_case(strip_fully_qualified_name(method.input_type))
        with Indenter(buf) as indent:
            indent.write('def {}(self, {}, **kwargs):\n'.format(method_name,
                                                                input_type))
            with Indenter(indent) as body_indent:
                write_method_body(body_indent, method)
    return buf.getvalue()


def generate_code(request, response):
    for proto_file in request.proto_file:
        buf = io.StringIO()
        buf.write(INTRO)
        head, sep, tail = proto_file.name.rpartition('.')
        _head = head.replace('/', '.')
        buf.write('import {}_pb2 as {}_pb2\n'.format(_head, _head.split('.')[-1]))
        buf.write('\n\n')
        f = response.file.add()
        f.name = head + '_stub.py'
        for service_descriptor in proto_file.service:
            buf.write(generate_service(service_descriptor))
        f.content = buf.getvalue()


def main():
    # Read request message from stdin
    data = sys.stdin.buffer.read()

    # Parse request
    request = plugin.CodeGeneratorRequest()
    request.ParseFromString(data)

    # Create response
    response = plugin.CodeGeneratorResponse()

    # Generate code
    generate_code(request, response)

    # Serialise response message
    output = response.SerializeToString()

    # Write to stdout
    sys.stdout.buffer.write(output)


if __name__ == '__main__':
    main()
