package main

import (
	pb "a.yandex-team.ru/infra/nanny/ictlstatus/proto"
	"bytes"
	"context"
	"fmt"
	"github.com/golang/protobuf/proto"
	"io/ioutil"
	"net"
	"net/http"
	"os"
	"time"
)

const (
	socketPath = "instancectl.sock"
	urlStatus  = "http://instancectl.sock/rpc/status/GetInstanceRevisionStatus/"
)

var (
	pbReq = pb.GetInstanceRevisionStatusRequest{}
)

func check(c *http.Client) (bool, error) {
	body, err := proto.Marshal(&pbReq)
	if err != nil {
		return false, fmt.Errorf("failed to marshall request: %v", err)
	}
	req, err := http.NewRequestWithContext(context.TODO(), "POST", urlStatus, bytes.NewReader(body))
	if err != nil {
		return false, err
	}
	req.Header.Add("Accept", "application/x-protobuf")
	req.Header.Add("Content-Type", "application/x-protobuf")
	req.Header.Add("User-Agent", "infra/ictl-status")
	resp, err := c.Do(req)
	if err != nil {
		return false, err
	}
	defer func() {
		_ = resp.Body.Close()
	}()
	if resp.StatusCode != 200 {
		return false, fmt.Errorf("non OK status code: %d", resp.StatusCode)
	}
	buf, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return false, fmt.Errorf("failed to read response: %v", err)
	}
	m := &pb.GetInstanceRevisionStatusResponse{}
	if err = proto.Unmarshal(buf, m); err != nil {
		return false, fmt.Errorf("failed to unmarshall response body: %v", err)
	}
	return m.Status != nil && m.Status.Ready != nil && m.Status.Ready.Status == "True", nil
}

func main() {
	defer func() {
		if r := recover(); r != nil {
			os.Exit(40)
		}
	}()
	c := &http.Client{
		Transport: &http.Transport{
			DialContext: func(ctx context.Context, _, _ string) (net.Conn, error) {
				d := net.Dialer{}
				return d.DialContext(ctx, "unix", socketPath)
			},
		},
		Timeout: 30 * time.Second,
	}
	defer c.CloseIdleConnections()
	deadline := time.Now().Add(10 * time.Minute)
	for time.Now().Before(deadline) {
		ok, err := check(c)
		if err != nil {
			fmt.Printf("Failed to check: %s\n", err)
		}
		if ok {
			fmt.Println("OK")
			os.Exit(0)
		} else {
			fmt.Println("instance is not ready")
		}
		time.Sleep(time.Second)
	}
	os.Exit(40)
}
