#!/usr/bin/env python
# coding: utf-8
from __future__ import print_function

import os
import sys
import errno
import argparse
import logging
import subprocess

import requests
from releaselib import setuppy


SVN_BASE_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia'
SVN_PROJECT_TAGS_URL_TEMPLATE = 'svn+ssh://arcadia.yandex.ru/arc/tags/infra/nanny/instancectl/{}'


def setup_logging():
    logger = logging.getLogger()
    console = logging.StreamHandler()
    console.setLevel(logging.INFO)
    logger.addHandler(console)
    logger.setLevel(logging.INFO)
    return logger


logger = setup_logging()


class SvnError(Exception):
    def __init__(self, message, exception=None):
        self.message = message
        self.exception = exception

    def __str__(self):
        rv = self.message
        e = self.exception
        if e:
            if isinstance(e, subprocess.CalledProcessError):
                rv += '\ncmd: {}\nreturncode: {}\noutput: {}'.format(e.cmd, e.returncode, e.output)
            else:
                rv += '\n{}'.format(e)
        return rv


def run(cmd, input_=None):
    if input_ is None:
        return subprocess.check_output(cmd, stderr=subprocess.STDOUT).strip()
    else:
        p = subprocess.Popen(cmd, stdin=subprocess.PIPE, stderr=subprocess.STDOUT, stdout=subprocess.PIPE, shell=False)
        return p.communicate(input_)[0].strip()


def create_svn_tag(base_url, tag_url):
    msg = 'Create tag {} from svn url {}'.format(tag_url, base_url)
    cmd = ['svn', 'copy', base_url, tag_url, '-m', msg]
    logger.debug(msg)
    try:
        run(cmd)
    except (subprocess.CalledProcessError, OSError) as e:
        raise SvnError('Failed to create tag', e)


def create_svn_commit(message):
    cmd = ['svn', 'commit', '-m', message]
    logger.debug('Creating svn commit...')
    try:
        r = run(cmd)
    except (subprocess.CalledProcessError, OSError) as e:
        raise SvnError('Failed to create commit', e)
    return r.split()[-1].rstrip('.')


def get_sandbox_oauth_token():
    try:
        with open(os.path.expanduser('~/.sandbox_oauth')) as fd:
            return fd.read().strip()
    except IOError as e:
        if errno.ENOENT == e.errno:
            return None


def ya_paste(content):
    try:
        with open('result.txt', 'w') as fp:
            fp.write(content)
        r = run(['ya', 'paste', '-t', 'diff', 'result.txt'])

        print(r)
    except (subprocess.CalledProcessError, OSError) as e:
        raise SvnError('Failed to paste patch', e)
    return r.split()[-1]


if __name__ == '__main__':
    token = get_sandbox_oauth_token()
    parser = argparse.ArgumentParser()
    parser.add_argument('--sandbox_oauth_token', default=token,
                        help='Sandbox OAuth token (will be taken from ~/.sandbox_oauth by default)')
    args = parser.parse_args()
    if args.sandbox_oauth_token is None:
        logger.error('Sandbox OAuth token not given and not found in ~/.sandbox_oauth. Please acquire it from https://sandbox.yandex-team.ru/oauth/.')
        sys.exit(1)
    new_version = setuppy.set_or_bump_version(bump='minor')
    version_str = 'stable/{}'.format(new_version)
    logger.info('Commit new version {}'.format(version_str))
    rev = create_svn_commit('Bump infra/nanny/instancectl version to {} SKIP_CHECK'.format(version_str))
    logger.info('Committed revision {}'.format(rev))
    logger.info('Creating svn tag')
    create_svn_tag(SVN_BASE_URL, SVN_PROJECT_TAGS_URL_TEMPLATE.format(version_str))
    logger.info('Creating svn patch')
    with open('template.patch') as fd:
        template = fd.read()
    rendered = template.format(version=str(new_version))
    res = ya_paste(rendered)
    patch_url = '{}/text'.format(res)
    ctx = {
        'ref_id': str(new_version),
        'ref_sha': rev,
        'patch_url': patch_url,
        'sshd_resource_id': '331426451',
        'juggler_client_resource_id': '2457076064',
        'logrotate_binary_resource_id': '438651806',
        'iss_hook_status_binary_resource_id': '1528089131',
        'gdb_toolkit_resource_id': '1683953475',
    }
    session = requests.Session()
    session.headers = {
        "Authorization": "OAuth {}".format(args.sandbox_oauth_token)
    }
    logger.info('Creating Sandbox task')
    resp = session.post('https://sandbox.yandex-team.ru/api/v1.0/task', json={
        "context": ctx,
        "owner": "NANNY",
        "type": "BUILD_INSTANCE_CTL",
        "description": "instancectl: {}".format(new_version),
        "priority": {
            "class": "SERVICE",
            "subclass": "HIGH"
        },
    })
    resp.raise_for_status()
    task_id = resp.json()['id']
    logger.info('https://sandbox.yandex-team.ru/task/{}/view'.format(task_id))
    logger.info('Starting Sandbox task')
    resp = session.put('https://sandbox.yandex-team.ru/api/v1.0/batch/tasks/start', json=[str(task_id)])
    resp.raise_for_status()
    logger.info('OK')
