#include <Python.h>

#include <unistd.h>
#include <fcntl.h>
#include <libgen.h>
#include <string.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>

#define DUMP_BLOCK_SIZE 512
#define SYNC_BLOCK (1 << 20)

static PyObject *CoreWriterError;

/* Derived from porto core dumper implementation */

int __remove_oldest_dump(const char* dir_path, uint64_t ttl) {
    char candidate[256];
    struct dirent *de;
    DIR *dir;
    time_t now = time(NULL);
    time_t oldest = now;
    int removed = 0;

    if (!ttl)
        return 0;

    memset(candidate, 0, sizeof(candidate));

    int dirfd = open(dir_path, O_RDONLY | O_DIRECTORY | O_CLOEXEC | O_NOATIME);
    if (dirfd < 0)
        return 0;

    dir = fdopendir(dirfd);
    if (dir == NULL) {
        close(dirfd);
        return 0;
    }

    while (de = readdir(dir)) {
        struct stat st;

        if (!strcmp(de->d_name, ".") || !strcmp(de->d_name, ".."))
            continue;

        if (fstatat(dirfd, de->d_name, &st, AT_SYMLINK_NOFOLLOW) < 0) {
            close(dirfd);
            return 0;
        }

        if (!S_ISREG(st.st_mode) || st.st_mtime + (time_t)ttl > now)
            continue;

        if (oldest > st.st_mtime) {
            oldest = st.st_mtime;
            strcpy(candidate, de->d_name);
        }
    }

    if (strlen(candidate))
        removed = !unlinkat(dirfd, candidate, 0);

    (void)closedir(dir);
    (void)close(dirfd);

    return removed;
}

PyObject *remove_oldest_dump(PyObject* self, PyObject* args) {
    char* dir;
    uint64_t ttl;

    Y_UNUSED(self);

    if (!PyArg_ParseTuple(args, "sK", &dir, &ttl)) {
        return NULL;
    }

    return Py_BuildValue("K", __remove_oldest_dump(dir, ttl));
}

PyObject *count_regular_files(PyObject* self, PyObject* args) {
    char *dir_path;

    Y_UNUSED(self);

    if (!PyArg_ParseTuple(args, "s", &dir_path))
        return PyErr_Format(CoreWriterError, "Bad arguments supplied");

    uint64_t count = 0, total_size = 0;
    struct dirent *de;
    DIR *dir;

    int dirfd = open(dir_path, O_RDONLY | O_DIRECTORY | O_CLOEXEC | O_NOATIME);
    if (dirfd < 0)
        return PyErr_Format(CoreWriterError, "Cannot open directory %s : %s",
                            dir_path, strerror(errno));

    dir = fdopendir(dirfd);
    if (dir == NULL) {
        int tmp = errno;
        close(dirfd);
        return PyErr_Format(CoreWriterError, "Cannot open dirent: %s",
                            strerror(tmp));
    }

    while (de = readdir(dir)) {
        if (!strcmp(de->d_name, ".") || !strcmp(de->d_name, ".."))
            continue;

        if (de->d_type == DT_REG) {
            struct stat st;

            if (fstatat(dirfd, de->d_name, &st, AT_SYMLINK_NOFOLLOW) < 0) {
                if (errno == ENOENT)
                    continue;

                int tmp = errno;

                close(dirfd);
                return PyErr_Format(CoreWriterError, "Cannot stat file %s: %s",
                                    de->d_name, strerror(tmp));
            }

            count++;
            /* Files can contain holes, thus we count number of blocks, instead of st_size */
            total_size += (st.st_blocks * 512) / (st.st_nlink ? st.st_nlink : 1);
        }
    }

    (void)closedir(dir);
    (void)close(dirfd);

    return Py_BuildValue("KK", count, total_size);
}

PyObject *save_core(PyObject* self, PyObject* args) {
    uint64_t buf[512];
    off_t size = 0lu;
    off_t sync_start = 0;
    int fd = -1, core_fd = -1;
    char* output_path;
    uint64_t ttl;

    Y_UNUSED(self);

    if (!PyArg_ParseTuple(args, "isK", &fd, &output_path, &ttl))
        return PyErr_Format(CoreWriterError, "Bad arguments supplied");

    if (fd < 0)
        return PyErr_Format(CoreWriterError, "Invalid input fd supplied");

    if (!output_path)
        return PyErr_Format(CoreWriterError, "Empty output path supplied");

    core_fd = open(output_path, O_RDWR | O_CREAT | O_EXCL | O_CLOEXEC, 0660);
    if (core_fd < 0)
        return PyErr_Format(CoreWriterError, "Cannot open output file %s : %s",
                            output_path, strerror(errno));

    do {
        size_t len = 0;

        do {
            ssize_t ret = read(fd, (uint8_t *)buf + len, sizeof(buf) - len);
            if (ret <= 0) {
                if (ret < 0) {
                    close(core_fd);
                    return PyErr_Format(CoreWriterError, "Cannot read from input fd: %s",
                                        strerror(errno));
                }

                break;
            }

            len += ret;
        } while (len < sizeof(buf));

        int zero = 1;

        for (size_t i = 0; zero && i < sizeof(buf) / sizeof(buf[0]); i++)
            zero = !buf[i];

        if (zero && len == sizeof(buf)) {
            size += len;
            continue;
        }

        size_t off = 0;
        int try_count = 0;

        while (off < len) {
            ssize_t ret = pwrite(core_fd, (uint8_t*)buf + off, len - off, size + off);
            if (ret <= 0) {
                if (ret < 0) {
                    if (!try_count &&
                        (errno == ENOSPC || errno == EDQUOT) &&
                        __remove_oldest_dump(dirname(output_path), ttl)) {
                        try_count++;
                        continue;
                    }

                    int tmp = errno;

                    close(core_fd);
                    return PyErr_Format(CoreWriterError, "Cannot write block: %s", strerror(tmp));
                }

                return PyErr_Format(CoreWriterError, "Zero bytes written instead of %lu", len - off);
            }

            try_count = 0;
            off += ret;
        }

        size += off;

        if (size > sync_start + 2 * SYNC_BLOCK) {
            (void)sync_file_range(core_fd, sync_start, size - sync_start,
                SYNC_FILE_RANGE_WAIT_BEFORE | SYNC_FILE_RANGE_WRITE);

            sync_start = size - SYNC_BLOCK;
        }

        if (off < len || !len)
            break;
    } while (1);

    if (ftruncate(core_fd, size)) {
        close(core_fd);
        return PyErr_Format(CoreWriterError, "Cannot truncate output file to size: %ld", size);
    }

    fdatasync(core_fd);

    return Py_BuildValue("k", size);
}

static PyMethodDef CoreWriterFunctions[] = {
    {"save_core", (PyCFunction)save_core, METH_VARARGS,
     "save core dump file consuming minimum resources"},
    {"remove_oldest_dump", (PyCFunction)remove_oldest_dump, METH_VARARGS,
     "remove dump files older than ttl secs"},
    {"count_regular_files", (PyCFunction)count_regular_files, METH_VARARGS,
     "count regular files reside in path"},
     {NULL, NULL, 0, NULL}};

PyMODINIT_FUNC initcorewriter() {
    PyObject *m = Py_InitModule("corewriter", CoreWriterFunctions);

    CoreWriterError = PyErr_NewException((char *)"corewriter.error", NULL, NULL);
    Py_INCREF(CoreWriterError);
    PyModule_AddObject(m, "error", CoreWriterError);
}
