from __future__ import unicode_literals

import logging
import os
import socket
from logging import handlers

import flask
from sepelib.util import fs
import gevent.pool
import gevent.queue
import gevent.event
from gevent import pywsgi

import inject
from instancectl import constants
from instancectl import utils
from instancectl.lib import unixsocket
from instancectl.rpc import blueprints
from instancectl.jobs import controller
from instancectl.status import interfaces as status_interfaces
from . import interfaces


class StartAction(interfaces.IInstanceCtlAction):

    log = logging.getLogger('actions.start')

    @staticmethod
    def create_unix_socket(path):
        # Assuming loop.lock is acquired, so there is no other instancectl instances
        # listen on instancectl.sock, so we can remove it

        # We have to create unixsocket then set 777 permissions because
        # instancectl may be started as root (for docker services)
        # and in this case ISS-agent (started as loadbase) cannot read from unixsocket
        # having "Permission denied" error
        temp_path = 'instancectl.sock.tmp'
        fs.remove_ignore(temp_path)
        s = socket.socket(socket.AF_UNIX)
        fs.set_close_exec(s.fileno())
        s.bind(temp_path)
        os.chmod(temp_path, 0o777)
        os.rename(temp_path, path)
        s.listen(1024)
        return s

    @classmethod
    def create_web_server(cls, socket_path):
        """
        :rtype: pywsgi.WSGIServer
        """
        app = flask.Flask('instancectl')
        for blueprint in blueprints:
            app.register_blueprint(blueprint)
        sock = cls.create_unix_socket(socket_path)
        pool = gevent.pool.Pool(size=constants.WEB_SERVER_POOL_SIZE)
        access_log = cls.make_access_log()
        return pywsgi.WSGIServer(listener=sock,
                                 application=app,
                                 handler_class=unixsocket.UnixsocketWSGIHandler,
                                 log=access_log,
                                 spawn=pool)

    @staticmethod
    def make_access_log():
        handler = handlers.RotatingFileHandler(constants.ACCESS_LOG_FILENAME,
                                               maxBytes=constants.LOG_MAX_BYTES,
                                               backupCount=constants.LOG_BACKUP_COUNT)
        return utils.StreamToHandler(handler)

    def __init__(self, lock_file, job_ctrl, status_cacher, status_updater, hq_status_reporter, stdout_rotater):
        """
        :type lock_file: file
        :type job_ctrl: instancectl.jobs.controller.JobController
        :type status_cacher: instancectl.status.cacher.InstanceRevisionStatusCacher
        :type status_updater: instancectl.status.updater.instance.InstanceRevisionStatusUpdater
        :type hq_status_reporter: instancectl.hq.reporter.HqInstanceStatusReporter
        :type stdout_rotater: instancectl.jobs.stdout_rotater.StdoutRotater
        """
        self.lock_file = lock_file
        self.status_cacher = status_cacher
        self.job_ctrl = job_ctrl
        self.status_updater = status_updater
        self.hq_status_reporter = hq_status_reporter
        self.stdout_rotater = stdout_rotater
        self.web = self.create_web_server(constants.UNIXSOCKET_PATH)
        inject.configure(self.inject_configure)

    def inject_configure(self, binder):
        """
        :type binder: inject.Binder
        """
        binder.bind(status_interfaces.IInstanceRevisionStatusCacher, self.status_cacher)
        binder.bind(controller.IJobController, self.job_ctrl)

    def run(self):
        self.status_updater.start()
        self.hq_status_reporter.start()
        # We must start server only after jobs inited.
        # It is necessary because in job init:
        # * We populate process env with secrets
        # * We open run.flag which MUST be passed to scripts
        #   (like stop_script, reopenlog_script, etc) to be killed by fuser in bsconfig
        self.job_ctrl.start()
        self.web.start()
        self.stdout_rotater.start()
        self.job_ctrl.wait()

    def stop(self):
        self.status_updater.stop()
        self.hq_status_reporter.stop()
        self.status_updater.update_instance_status()
        self.hq_status_reporter.report()
        self.stdout_rotater.stop()
        self.web.stop()
