import socket

from requests.adapters import HTTPAdapter
from requests.compat import urlparse, unquote
from requests.packages.urllib3 import exceptions
from requests.packages.urllib3.connection import HTTPConnection
from requests.packages.urllib3.connectionpool import HTTPConnectionPool

from . import exceptions as socket_exceptions


# The following was adapted from some code from docker-py
# https://github.com/docker/docker-py/blob/master/docker/unixconn/unixconn.py
class UnixHTTPConnection(HTTPConnection):

    def __init__(self, unix_socket_url, *args, **kwargs):
        """Create an HTTP connection to a unix domain socket
        :param unix_socket_url: A URL with a scheme of 'http+unix' and the
        netloc is a percent-encoded path to a unix domain socket. E.g.:
        'http+unix://%2Ftmp%2Fprofilesvc.sock/status/pid'
        """
        HTTPConnection.__init__(self, 'localhost', *args, **kwargs)
        self.socket_path = unquote(urlparse(unix_socket_url).netloc)

    def connect(self):

        sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        try:

            if self.timeout is not socket._GLOBAL_DEFAULT_TIMEOUT:
                sock.settimeout(self.timeout)

            sock.connect(self.socket_path)

        except socket.timeout as e:
            sock.close()
            raise exceptions.ConnectTimeoutError(
                self, "Connection to %s timed out. (connect timeout=%s)" %
                      (self.host, self.timeout))

        except socket.error as e:
            sock.close()
            raise socket_exceptions.ConnectionError(
                "Failed to establish a new connection: %s" % e)

        self.sock = sock


class UnixHTTPConnectionPool(HTTPConnectionPool):

    def __init__(self, url):
        HTTPConnectionPool.__init__(self, 'localhost')
        self.url = url

    def _new_conn(self):
        return UnixHTTPConnection(self.url)


class UnixAdapter(HTTPAdapter):

    def get_connection(self, url, proxies=None):
        proxies = proxies or {}
        proxy = proxies.get(urlparse(url.lower()).scheme)

        if proxy:
            raise ValueError('{} does not support specifying proxies'.format(self.__class__.__name__))

        return UnixHTTPConnectionPool(url)


class UnixHttp1HTTPConnection(UnixHTTPConnection):
    _http_vsn = 10
    _http_vsn_str = 'HTTP/1.0'


class UnixHttp1HTTPConnectionPool(UnixHTTPConnectionPool):

    def _new_conn(self):
        return UnixHttp1HTTPConnection(self.url)


class UnixHttp1Adapter(HTTPAdapter):

    def get_connection(self, url, proxies=None):
        proxies = proxies or {}
        proxy = proxies.get(urlparse(url.lower()).scheme)

        if proxy:
            raise ValueError('{} does not support specifying proxies'.format(self.__class__.__name__))

        return UnixHttp1HTTPConnectionPool(url)
