from __future__ import  unicode_literals

import requests

from instancectl.lib.rpc import exceptions
from instancectl.lib.rpc import status_pb2
from . import adapters


UNIX_SOCKET_SCHEME = 'http+unix://'


def make_exception_from_pb(error):
    exc_class = exceptions.errors_by_code.get(error.code)
    if exc_class is None:
        raise RuntimeError('Unsupported response code for error: {}'.format(error))
    return exc_class(error.message, redirect_url=error.redirect_url)


class UnixsocketRpcClient(object):
    """
    Implementation of HTTP RPC client, which uses requests library to perform calls.
    """
    DEFAULT_HEADERS = {
        'Accept': 'application/x-protobuf',
        'Content-Type': 'application/x-protobuf',
        'User-Agent': 'Python UnixsocketRpcClient'
    }

    def __init__(self, rpc_url, request_timeout=None):
        self._rpc_url = rpc_url
        self._request_timeout = request_timeout
        self._session = requests.Session()
        self._session.mount(UNIX_SOCKET_SCHEME, adapters.UnixAdapter())

    # ########### Interface implementation
    def call_remote_method(self, method_name, request_protobuf, response_protobuf, **kwargs):
        url = '{0}/{1}/'.format(self._rpc_url, method_name)
        headers = self.DEFAULT_HEADERS.copy()
        kwargs = {
            'allow_redirects': False,
            'data': request_protobuf.SerializeToString(),
            'headers': headers,
            'timeout': kwargs.get('request_timeout', self._request_timeout)
        }

        response = self._session.post(url, **kwargs)
        if response.status_code == 200:  # Response is ok, deserialize
            response_protobuf.MergeFromString(response.content)
            return
        # Error happened, read from body and raise exception
        error_protobuf = status_pb2.Status()
        try:
            error_protobuf.MergeFromString(response.content)
        except Exception:
            # Failed to load protobuf error response
            raise RuntimeError('Invalid response from server: {0}'.format(response.content))
        raise make_exception_from_pb(error_protobuf)
