from __future__ import unicode_literals

import logging

import gevent
import json
import requests
from google.protobuf import json_format
from infra.nanny.instancectl.proto import secret_pb2
from library.python import resource
from sepelib.util import retry
from . import errors
from . import helpers


def _cast_secret_to_secret_revision_path(secret):
    """
    :type secret: clusterpb.types_pb2.KeychainSecret
    :rtype: unicode
    """
    return '{}/{}/{}'.format(secret.keychain_id, secret.secret_id, secret.secret_revision_id)


class VaultClient(object):
    RETRY_EXCEPTIONS = (
        requests.RequestException,
        gevent.Timeout,
    )

    ATTEMPT_TIMEOUT = 5.0

    def __init__(self, url, service_id, token=None):
        self.log = logging.getLogger(__name__ + '.' + self.__class__.__name__)
        self._url = url.rstrip('/')
        self._service_id = service_id
        if not token:
            token = resource.find('/secrets/nanny_vault_oauth_token').strip()
        self._token = token

    def _auth(self, secret):
        """
        :type secret: clusterpb.types_pb2.KeychainSecret
        :rtype: unicode
        """
        self.log.info('Auth in Nanny Vault')
        auth_url = '{}/v1/auth/blackbox/service/login/{}'.format(self._url, self._service_id)
        resp = requests.post(
            auth_url,
            headers={'Content-Type': 'application/json'},
            timeout=5,
            data=json.dumps({'keychain': secret.keychain_id,
                             'oauth': self._token})
        )
        if resp.status_code == 400:
            raise errors.VaultError('Failed to fetch keychain "{}" secret "{}" revision "{}": {} {}. Service is not '
                                    'allowed to read the keychain probably. Please add this service to the allowed '
                                    'services list on keychain "{}" page.'.format(secret.keychain_id,
                                                                                  secret.secret_id,
                                                                                  secret.secret_revision_id,
                                                                                  resp.status_code,
                                                                                  resp.reason,
                                                                                  secret.keychain_id))
        resp.raise_for_status()
        return resp.json()['auth']['client_token']

    def _get(self, token, secret):
        self.log.info('Getting secret from Nanny Vault')
        url = '{}/v1/secret/{}'.format(self._url, _cast_secret_to_secret_revision_path(secret))
        try:
            resp = requests.get(url,
                                headers={'X-Vault-Token': token},
                                timeout=5)
        except Exception as e:
            raise errors.VaultError("Failed to fetch secret from Nanny Vault (request failed): {}".format(e))

        if resp.status_code == 404:
            raise errors.VaultError('Failed to fetch keychain "{}" secret "{}" revision "{}": desired secret or '
                                    'its revision not found: {} {}'.format(secret.keychain_id,
                                                                           secret.secret_id,
                                                                           secret.secret_revision_id,
                                                                           resp.status_code,
                                                                           resp.reason))

        if resp.status_code != 200:
            raise errors.VaultError('Failed to fetch keychain "{}" secret "{}" revision "{}" (bad status code): '
                                    '{} {}'.format(secret.keychain_id,
                                                   secret.secret_id,
                                                   secret.secret_revision_id,
                                                   resp.status_code,
                                                   resp.reason))
        try:
            data = resp.json()
        except Exception as e:
            raise errors.VaultError('Failed to fetch keychain "{}" secret "{}" revision "{}" (json parse failed): '
                                    '{}'.format(secret.keychain_id, secret.secret_id, secret.secret_revision_id, e))
        if 'data' not in data:
            raise errors.VaultError('Failed to fetch keychain "{}" secret "{}" revision "{}" (no "data" in '
                                    'response)'.format(secret.keychain_id, secret.secret_id, secret.secret_revision_id))
        return data['data']

    def get_secret(self, secret):
        """
        :type secret: clusterpb.types_pb2.KeychainSecret
        :rtype: dict
        """
        s = retry.RetrySleeper(max_tries=5, delay=1, backoff=3)
        r = retry.RetryWithTimeout(attempt_timeout=self.ATTEMPT_TIMEOUT,
                                   retry_sleeper=s,
                                   retry_exceptions=self.RETRY_EXCEPTIONS)
        vault_token = r(self._auth, secret)
        secret = r(self._get, vault_token, secret)
        m = secret_pb2.Secret()
        try:
            json_format.ParseDict(secret, m)
        except json_format.ParseError:
            # Could not parse secret, assuming it is old secret which has no schema
            return helpers.encode_legacy_secret_dict_values(secret)

        return helpers.encode_secret_entry_values(m)

    def get_secret_field(self, secret, field):
        """
        :type secret: clusterpb.types_pb2.KeychainSecret
        :type field: unicode
        :rtype: unicode
        """
        data = self.get_secret(secret)

        if not field:
            if len(data) > 1:
                raise errors.VaultError('Desired Nanny Vault secret contains multiple key-value pairs but instance spec'
                                        ' has no secret field defined to extract value from')
            if len(data) < 1:
                raise errors.VaultError('Desired Nanny Vault secret is empty')

            return data.values()[0]

        if field not in data:
            raise errors.VaultError('Desired Nanny Vault secret field "{}" not found in secret'.format(field))

        return data[field]
