from __future__ import unicode_literals

import time
import platform

import base64
import logging
import os

import gevent
import requests
import ticket_parser2
import ticket_parser2.low_level as tp2
from instancectl.version import VERSION
from library.python import resource
from sepelib.util import retry
from . import errors


class YavClient(object):
    RETRY_EXCEPTIONS = (
        requests.RequestException,
        gevent.Timeout,
    )

    ATTEMPT_TIMEOUT = 5.0

    def __init__(self, url, service_id, tvm_client_id, yav_tvm_client_id, tvm_api_url):
        self.log = logging.getLogger(__name__ + '.' + self.__class__.__name__)
        self._url = url.rstrip('/')
        self._service_id = service_id
        self._tvm_client_id = tvm_client_id
        self._yav_tvm_client_id = yav_tvm_client_id
        self._tvm_api_url = tvm_api_url
        self._session = requests.Session()
        self._user_agent = 'Instancectl {ictl_ver} ({system}; {platform}; {node}) {p_impl}/{p_ver} ({p_comp})'.format(
            ictl_ver=VERSION,
            system=platform.system(),
            platform=platform.platform(),
            node=platform.node(),
            p_impl=platform.python_implementation(),
            p_ver=platform.python_version(),
            p_comp=platform.python_compiler(),
        )
        self._session.headers = {
            'User-Agent': self._user_agent
        }

    def _get_yav_ticket(self):
        client_secret = resource.find('/secrets/nanny_tvm_secret').strip()
        if not client_secret and os.getenv('NANNY_TVM_SECRET') is not None:
            client_secret = os.getenv('NANNY_TVM_SECRET')
        self.log.info('Getting TVM service ticket to YaVault')
        tvm_keys_resp = self._session.get(
            '{}/2/keys?lib_version={version}'.format(self._tvm_api_url, version=ticket_parser2.__version__)
        )
        if tvm_keys_resp.status_code != 200:
            raise errors.YavError("Couldn't fetch public TVM keys")
        tvm_keys = tvm_keys_resp.content
        sc = tp2.ServiceContext(int(self._tvm_client_id), client_secret, tvm_keys.decode())
        ts = str(int(time.time()))
        tickets_resp = self._session.post('{}/2/ticket/'.format(self._tvm_api_url), data={
            'grant_type': 'client_credentials',
            'src': int(self._tvm_client_id),
            'dst': int(self._yav_tvm_client_id),
            'ts': ts,
            'sign': sc.sign(ts, str(self._yav_tvm_client_id), ''),
        })
        if tickets_resp.status_code != 200:
            raise errors.YavError("Failed to get TVM service ticket to YaVault: {}".format(tickets_resp.status_code))
        tickets = tickets_resp.json()
        if str(self._yav_tvm_client_id) not in tickets:
            raise errors.YavError("Failed to get TVM service ticket to YaVault: {}".format(tickets_resp.status_code))
        return tickets[str(self._yav_tvm_client_id)]['ticket']

    def _get(self, service_ticket, secret):
        self.log.info('Getting secret from YaVault')
        try:
            resp = self._session.post('{}/1/tokens?consumer=instancectl.{}'.format(self._url, VERSION),
                                      json={'tokenized_requests': [
                                          {
                                              'token': secret.delegation_token,
                                              'signature': self._service_id,
                                              'secret_version': secret.secret_ver,
                                              'service_ticket': service_ticket
                                          }
                                      ]},
                                      timeout=5
                                      )
        except Exception as e:
            raise errors.YavError("Failed to fetch YaVault secret (request failed): {}".format(e))
        try:
            data = resp.json()
        except Exception as e:
            raise errors.YavError('Failed to fetch YaVault secret "{}" revision "{}" (json parse failed): {}'.format(
                secret.secret_id, secret.secret_ver, e))
        if resp.status_code != 200:
            raise errors.YavError('Failed to fetch YaVault secret "{}" version "{}": desired secret or '
                                  'its version not found: {}'.format(secret.secret_id,
                                                                     secret.secret_ver,
                                                                     resp.status_code))
        if 'secrets' not in data:
            raise errors.YavError(
                'Failed to fetch YaVault secret "{}" version "{}" (no "secrets" in response)'.format(
                    secret.secret_id, secret.version_id))
        if 'status' in data and data['status'] != 'ok':
            raise errors.YavError(
                'Bad status in YaVault response for secret "{}" version "{}"'.format(
                    secret.secret_id, secret.version_id))
        return data['secrets']

    def get_secret(self, secret):
        """
        :type secret: clusterpb.types_pb2.VaultSecret
        :rtype: dict
        """
        s = retry.RetrySleeper(max_tries=5, delay=1, backoff=3)
        r = retry.RetryWithTimeout(attempt_timeout=self.ATTEMPT_TIMEOUT,
                                   retry_sleeper=s,
                                   retry_exceptions=self.RETRY_EXCEPTIONS)
        yav_ticket = r(self._get_yav_ticket)
        secrets = r(self._get, yav_ticket, secret)

        rv = {}
        for s in secrets:
            if 'status' in s and s['status'] != 'ok':
                raise errors.YavError(
                    'Bad status in YaVault response for secret "{}" version "{}"'.format(
                        secret.secret_id, secret.secret_ver))
            for field in s['value']:
                value = field['value']
                encoding = field.get('encoding', None)
                if encoding == 'base64':
                    value = base64.b64decode(value)
                rv[field['key']] = value
        return rv

    def get_secret_field(self, secret, field):
        """
        :type secret: clusterpb.types_pb2.VaultSecret
        :type field: unicode
        :rtype: unicode
        """
        data = self.get_secret(secret)

        if not field:
            if len(data) > 1:
                raise errors.YavError('Desired YaVault secret contains multiple key-value pairs but instance spec '
                                      'has no secret field defined to extract value from')
            if len(data) < 1:
                raise errors.YavError('Desired YaVault secret is empty')

            return data.values()[0]

        if field not in data:
            raise errors.YavError('Desired YaVault secret field "{}" not found in secret'.format(field))

        return data[field]
