from __future__ import unicode_literals

import atexit
import logging
import os
import sys
import corewriter
import random
import time

from instancectl.lib import coreaggrutil
from instancectl.lib import minidumputil

from instancectl.config import defaults

CORE_PROC_STDOUT_FILENAME = 'core_process.out'
CORE_PROC_STDERR_FILENAME = 'core_process.err'

log = logging.getLogger('actions.core_process')


def _ensure_limits(core_dir, args, need_core_space):
    reg_num, _ = corewriter.count_regular_files(core_dir)

    _count_needed = args.count_limit - (1 if need_core_space else 0)

    # Carefully clean some old dumps
    if _count_needed > 0 and reg_num > _count_needed:

        for count in xrange(reg_num, _count_needed, -1):
            if corewriter.remove_oldest_dump(core_dir, args.ttl) == 0:
                log.error("Cannot overcome core count limit, no files to remove")
                return False

            cur_reg, _ = corewriter.count_regular_files(core_dir)

            if cur_reg >= count:
                log.error("Cannot overcome core count limit, racing with someone")
                return False

            # It's a miracle, bail out ASAP
            if cur_reg < args.count_limit:
                break

    _, coredump_size = corewriter.count_regular_files(core_dir)

    # Ok, we're exceeding total size limit, where's my axe?
    while args.total_size_limit > 0 and coredump_size >= args.total_size_limit:
        if corewriter.remove_oldest_dump(core_dir, args.ttl) == 0:
            log.error("Cannot overcome total core size limit, no files to remove")
            return False

        _, new_size = corewriter.count_regular_files(core_dir)

        if new_size > coredump_size:
            log.error("Cannot overcome total core size limit, racing with someone")
            return False

        coredump_size = new_size

    return True


def ensure_limits(core_dir, args, need_core_space=False):
    try:
        return _ensure_limits(core_dir, args, need_core_space)
    except Exception as e:
        log.error("Error while checking limits : {}".format(e), exc_info=True)
        return False


def send_traces(output_path, args):

    if not (args.gdb_path and args.aggr_url and args.aggr_svc_name and args.aggr_instance_name):
        log.info("Not enough aggregation settings specified, skipping")
        return

    if not os.access(args.gdb_path, os.X_OK):
        log.info("No executable gdb found in path %s", args.gdb_path)
        return

    binary_path = os.readlink("/proc/{}/exe".format(os.environ['CORE_PID']))
    parse_result = coreaggrutil.ParseResult.OK
    parse_error = ""
    parsed_traces = ""

    try:
        parsed_traces = minidumputil.extract_coredump_traces(
            file_name=output_path,
            binary=binary_path,
            gdb_path=args.gdb_path,
            gdb_timeout=int(defaults.optional_defaults.get('coredumps_gdb_stackwalk_timeout', 120)),
            limits={}
        )
    except Exception as err:
        msg = "Cannot parse coredump traces: {}".format(err)
        log.error(msg, exc_info=True)
        parse_result = coreaggrutil.ParseResult.FAILED
        if not coreaggrutil.is_new_cores_aggregator(args.aggr_url):
            return
        parse_error = msg

    if not parsed_traces and not coreaggrutil.is_new_cores_aggregator(args.aggr_url):
        log.error("Got empty traces from coredump")
        return

    try:
        minidumputil.send_minidump_to_saas(
            file_name=output_path,
            parsed_traces=parsed_traces,
            url=args.aggr_url,
            service=args.aggr_svc_name,
            instance_name=args.aggr_instance_name,
            ctype=args.aggr_ctype,
            parse_error=parse_error,
            parse_result=parse_result,
        )

    except Exception as e:
        log.error("Cannot send minidump info to SAAS: {}".format(e), exc_info=True)


def core_proc_action(args, console_handler):
    """
    Receives coredump from stdin, saves, process and hands it to core aggregator.
    """
    #   FIXME: following fails with stdout=/dev/null and stderr=/dev/null
    #   setup_logging.setup_logging(console_handler, False)

    sys.stdout.flush()
    sys.stderr.flush()

    with open(CORE_PROC_STDOUT_FILENAME, 'a') as fd:
        os.dup2(fd.fileno(), sys.stdout.fileno())

    with open(CORE_PROC_STDERR_FILENAME, 'a') as fd:
        os.dup2(fd.fileno(), sys.stderr.fileno())

    log.info("Catching coredump")

    @atexit.register
    def log_exit():
        log.info("Exited")

    if random.randint(0, 99) >= args.core_probability:
        log.info("No core for you at the moment, pal. Probability is: {}".format(args.core_probability))
        return

    # Since coredumps can be persistent, the pids can be easily reused on restarts,
    # so let's append timestamp suffix by ourselves, since porto doesn't provide
    # env var for it
    output_path = "{}-{}".format(args.output_path, int(time.time() * 1000))

    core_dir = os.path.dirname(args.output_path)

    if not ensure_limits(core_dir, args, need_core_space=True):
        return

    try:
        ret = corewriter.save_core(0, output_path, args.ttl)
        if ret == 0:
            raise Exception("Nothing written so far")
        else:
            log.info("Wrote {} coredump, size: {} bytes"
                     .format(output_path, ret))

    except Exception as e:
        log.error("Error while writing core dump {} : {}"
                  .format(output_path, e), exc_info=True)
        os.unlink(output_path)
        return

    send_traces(output_path, args)

    ensure_limits(core_dir, args)


def add_parsers(subparsers):
    parser = subparsers.add_parser(
        name='core_process',
        description='Coredump catcher helper'
    )
    parser.set_defaults(handle=core_proc_action)
    parser.add_argument(
        "-o", "--output", metavar="CORE_OUTPUT_PATH", dest="output_path",
        help="core output file path", type=str
    )
    parser.add_argument(
        "-c", "--count-limit", metavar="CORE_COUNT_LIMIT", dest="count_limit",
        help="core files count limit", type=int, default=0
    )
    parser.add_argument(
        "-b", "--total-size-limit", metavar="CORE_SIZE_LIMIT", dest="total_size_limit",
        help="stored core files total limit", type=int, default=0
    )
    parser.add_argument(
        "-p", "--probability", metavar="CORE_PROBABILITY", dest="core_probability",
        help="Store core file with specified probability", type=int, default=100
    )
    parser.add_argument(
        "-t", "--ttl", metavar="CORE_TTL", dest="ttl",
        help="Remove files older than ttl secs if needed", type=int, default=0
    )
    parser.add_argument(
        "--aggr-url", metavar="CORE_REPORT_URL", dest="aggr_url",
        help="Post stacktrace to by this url", type=str,
        default=defaults.optional_defaults.get('minidumps_aggregator_url', "")
    )
    parser.add_argument(
        "--svc-name", metavar="CORE_REPORT_SVC_NAME", dest="aggr_svc_name",
        help="Service name to use for saas crash reporting", type=str,
        default=defaults.optional_defaults.get('minidumps_service', "")
    )
    parser.add_argument(
        "--instance-name", metavar="CORE_REPORT_INSTANCE_NAME", dest="aggr_instance_name",
        help="Instance name to use for saas crash reporting", type=str,
        default=os.environ.get('BSCONFIG_INAME', "")
    )
    parser.add_argument(
        "--ctype", metavar="CORE_REPORT_CTYPE", dest="aggr_ctype",
        help="Cluster type to be sent in saas crash reporting", type=str,
        default=""
    )
    parser.add_argument(
        "--gdb-path", metavar="GDB_PATH", dest="gdb_path",
        help="Path to gdb executable, default: /usr/bin/gdb", type=str,
        default=defaults.optional_defaults.get('./gdb/bin/gdb', "/usr/bin/gdb")
    )
