from __future__ import unicode_literals

import atexit
import logging
import sys

from instancectl import constants
from instancectl.clients.unixsocket_rpc import client
from instancectl.jobs.job import JobStatusCheckResult
from instancectl.status import external
from instancectl.utils import get_event_logger
from instancectl.cmd import log as setup_logging


log = logging.getLogger('actions.status')
event_log = get_event_logger('status')


def get_status(args, console_handler):
    """
    Gets instance status from unixsocket.

    Assuming that we are in the instance directory.
    """
    setup_logging.setup_logging(console_handler, args.console)

    log.info("Starting")

    @atexit.register
    def log_exit():
        log.info("Exited")

    url = '{}{}/rpc/status'.format(client.UNIX_SOCKET_SCHEME, constants.UNIXSOCKET_PATH)
    checker = external.ExternalInstanceStatusChecker(
        url=url,
        max_tries=args.max_tries,
        delay=args.min_delay,
        backoff=args.delay_backoff,
        max_delay=args.max_delay,
        req_timeout=args.req_timeout,
    )
    try:
        s = checker.get_instance_status()
    except external.InstanceNotReadyError as e:
        s = e.status
    except Exception:
        log.critical('Cannot check instance status', exc_info=True)
        event_log.exception('Cannot check instance status')
        # Never return NOT_STARTED even if status getting failed,
        # otherwise ISS will stop instance and reinstall it
        sys.exit(JobStatusCheckResult.PENDING)

    if s.ready.status == constants.CONDITION_TRUE:
        sys.exit(JobStatusCheckResult.STARTED)
    event_log.info('Instance status check result: FAIL')
    log.info('Status check result: FAIL')
    sys.exit(JobStatusCheckResult.PENDING)


def add_parsers(subparsers):
    parser = subparsers.add_parser('status', description='Get instance status code')
    parser.set_defaults(handle=get_status)
    parser.add_argument(
        "-c", "--config", metavar="CONFIGFILE", dest="config",
        help="config file", type=str, default='loop.conf',
    )
    parser.add_argument(
        "--max-tries", metavar="STATUS_CHECK_MAX_TRIES", dest="max_tries",
        help="Attempts to get instance status", type=int, default=constants.STATUS_CHECK_TRIES,
    )
    parser.add_argument(
        "--min-delay", metavar="STATUS_CHECK_MIN_DELAY", dest="min_delay",
        help="Initial delay between status check attempts", type=float, default=constants.STATUS_CHECK_MIN_DELAY,
    )
    parser.add_argument(
        "--max-delay", metavar="STATUS_CHECK_MAX_DELAY", dest="max_delay",
        help="Initial delay between status check attempts", type=float, default=constants.STATUS_CHECK_MAX_DELAY,
    )
    parser.add_argument(
        "--delay-backoff", metavar="STATUS_CHECK_DELAY_BACKOFF", dest="delay_backoff",
        help="Backoff for delay between status check attempts", type=float, default=constants.STATUS_CHECK_BACKOFF,
    )
    parser.add_argument(
        "--req-timeout", metavar="STATUS_CHECK_REQ_TIMEOUT", dest="req_timeout",
        help="Timeout for the instance status request attempt", type=float,
        default=constants.STATUS_CHECK_REQ_TIMEOUT,
    )
