from __future__ import unicode_literals

import sys
import logging
from logging import handlers

import its_client.poller
from sepelib.util.log import setup_logging_to_file
from sepelib.util.log import utils

from instancectl import constants
from instancectl.utils import EVENTS_LOGGER_NAME


EVENT_LOG_PATH = 'loop.log'
FULL_LOG_PATH = 'loop.log.full'
ITS_CLIENT_LOG_PATH = 'its_client.log'
LOG_FORMAT = "%(hostname)s %(process)s %(asctime)s %(levelname)s - %(name)s - %(message)s"


def setup_logging(console_handler, console=False):
    """
    Setup service logs

    :type console_handler: logging.Handler
    :type console: bool
    """
    # disable requests module logging
    logging.getLogger('requests').setLevel(logging.WARNING)

    if sys.stdout.isatty() or console:
        console_handler.setLevel(logging.DEBUG)
        return

    # Common logger
    handler = _create_log_handler(FULL_LOG_PATH)
    logger = setup_logging_to_file(handler,
                                   redirect_stderr=False,
                                   fmt=LOG_FORMAT)
    logging.getLogger().removeHandler(console_handler)

    # Output CRITICAL to stderr: SWAT-1924
    formatter = logging.Formatter(LOG_FORMAT)
    stderr_handler = logging.StreamHandler(sys.stderr)
    stderr_handler.setLevel(logging.CRITICAL)
    stderr_handler.setFormatter(formatter)
    logger.addHandler(stderr_handler)

    # Important events logger: SWAT-2199
    handler = _create_log_handler(EVENT_LOG_PATH)
    setup_logging_to_file(handler,
                          redirect_stdout=False,
                          redirect_stderr=False,
                          logger_name=EVENTS_LOGGER_NAME,
                          fmt=LOG_FORMAT)

    # Log ITS actions to separate file: SWAT-2464
    handler = _create_log_handler(ITS_CLIENT_LOG_PATH)
    setup_logging_to_file(handler,
                          redirect_stdout=False,
                          redirect_stderr=False,
                          logger_name=its_client.poller.LOGGER_NAME)


def _create_log_handler(log_path):
    return handlers.RotatingFileHandler(log_path,
                                        maxBytes=constants.LOG_MAX_BYTES,
                                        backupCount=constants.LOG_BACKUP_COUNT)


def rename_loglevels():
    """
    Rename log levels so that all have the same length of 5 characters.
    This makes them look nice and consistent in log file.
    """
    short_levelnames = {logging.WARN: "WARN ",
                        logging.INFO: "INFO "}
    for k, v in short_levelnames.iteritems():
        logging.addLevelName(k, v)


def setup_logging_to_stderr():
    """
    Setup logging handler pushing everything into stdout.
    :rtype: logging.Handler
    """
    rename_loglevels()
    formatter = utils.AugmentedFormatter(LOG_FORMAT)

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    stdout_handler = logging.StreamHandler(sys.stderr)
    stdout_handler.setLevel(logging.DEBUG)
    stdout_handler.setFormatter(formatter)
    logger.addHandler(stdout_handler)

    return stdout_handler
