from __future__ import unicode_literals

import json
import logging
import sys

from google.protobuf import json_format

from instancectl import constants
from instancectl.clients.unixsocket_rpc import client
from instancectl.jobs.job import JobStatusCheckResult
from instancectl.status import external


log = logging.getLogger('actions.status')


def print_status(args, console_handler):
    """
    Gets instance status from unixsocket.

    Assuming that we are in the instance directory.
    """
    console_handler.setLevel(logging.WARNING)

    url = '{}{}/rpc/status'.format(client.UNIX_SOCKET_SCHEME, constants.UNIXSOCKET_PATH)
    checker = external.ExternalInstanceStatusChecker(
        url=url,
        max_tries=1,
        delay=0,
        backoff=1,
        max_delay=1,
        req_timeout=args.req_timeout,
    )
    try:
        s = checker.get_instance_status()
    except external.InstanceNotReadyError as e:
        s = e.status
    except Exception:
        log.critical('Cannot check instance status', exc_info=True)
        sys.exit(JobStatusCheckResult.PENDING)
    ready = s.ready.status == constants.CONDITION_TRUE

    if args.format == constants.STATUS_FORMAT_JSON:
        print_status_json(s)
    elif args.format == constants.STATUS_FORMAT_HUMAN:
        print_status_human(s)
    if ready:
        sys.exit(JobStatusCheckResult.STARTED)
    sys.exit(JobStatusCheckResult.PENDING)


def ascii_table(table):
    """
    Prepare table for printing

    1 2 3      4
    1 2 Hello  world
    2 2 foo    bar

    :type table: list
    :rtype: unicode
    """
    max_f = 0
    for t in table:
        cnt_field = len(t)
        if max_f < cnt_field:
            max_f = cnt_field
        if cnt_field < max_f:
            for i in range(max_f-cnt_field):
                t.append("-")
    col_width = [max(len(str(x)) for x in col) for col in zip(*table)]
    ret_string = ""
    for line in table:
        ret_string += "  ".join("{:{}}".format(x, col_width[i])
                                for i, x in enumerate(line)) + "\n"
    return ret_string


def print_status_json(s):
    d = json_format.MessageToDict(s)
    print json.dumps(d, indent=4)


def print_status_human(s):
    table = [
        ['Name', 'Ready', 'Installed'],
        ['-----------', '-----------', '-----------'],
    ]
    for c in s.container:
        ready = 'OK' if c.ready.status == constants.CONDITION_TRUE else 'FAIL'
        installed = 'OK' if c.installed.status == constants.CONDITION_TRUE else 'FAIL'
        table.append([c.name, ready, installed])
    print ascii_table(table)


def add_parsers(subparsers):
    parser = subparsers.add_parser('s', description='Print instance status')
    parser.set_defaults(handle=print_status)
    parser.add_argument(
        "--req-timeout", metavar="STATUS_CHECK_REQ_TIMEOUT", dest="req_timeout",
        help="Timeout for the instance status request attempt", type=float,
        default=constants.STATUS_CHECK_REQ_TIMEOUT,
    )
    parser.add_argument(
        "--format", metavar="FORMAT", dest="format",
        help="Output format (default: human)", type=str, default=constants.STATUS_FORMAT_HUMAN,
        choices=(constants.STATUS_FORMAT_JSON, constants.STATUS_FORMAT_HUMAN)
    )
