from __future__ import unicode_literals

import atexit
import logging
import sys

from instancectl.config.config import get_instance_config
from instancectl.config.errors import InstanceCtlConfigError
from instancectl.jobs import job
from instancectl.jobs import helpers
from instancectl.lib import confutil
from instancectl.cmd import log as setup_logging


log = logging.getLogger('actions.send_minidump')


def send_minidump(args, console_handler):
    """
    Process minidump/coredump and sends result to cores aggregator.
    """
    setup_logging.setup_logging(console_handler, args.console)
    log.info("Sending minidump to aggregator")

    @atexit.register
    def log_exit():
        log.info("Minidump sending finished")

    try:
        config = get_instance_config()
    except InstanceCtlConfigError:
        log.critical('Instance config is not compiled', exc_info=True)
        sys.exit(job.JobStatusCheckResult.NOT_INSTALLED)

    if args.section not in config['jobs']:
        log.error('Could not find job for section %s', args.section)
        sys.exit(1)

    # FIXME: it must be done with call to main instancectl process on unixsocket
    c = config['jobs'][args.section]
    if not c['minidumps_push']:
        log.info('Minidumps pushing for section %s disabled, will not send minidump %s', args.section, args.filename)
        return

    limits = helpers.make_limits_dict_from_config(c)
    s = confutil.make_minidump_sender_from_config(args.section, c, limits)
    s.send_minidump(args.filename)


def add_parsers(subparsers):
    parser = subparsers.add_parser(name='send_minidump', description='Send minidump to aggregator')
    parser.set_defaults(handle=send_minidump)
    parser.add_argument("section", metavar="SECTION", help="loop.conf binary section name", type=str)
    parser.add_argument("filename", metavar="MINIDUMP_FILENAME", help="minidump filename", type=str)
