from __future__ import unicode_literals

import atexit
import logging

from instancectl.lib import actionsutil
from instancectl.cms import detect_cms_agent_type
from instancectl.utils import detach_from_cms_agent
from instancectl.cmd import log as setup_logging

log = logging.getLogger('actions.start')


def start_instance(args, console_handler):
    """
    Starts instance.

    Assuming that we are in the instance directory.
    """
    setup_logging.setup_logging(console_handler, args.console)
    log.info("Starting")

    @atexit.register
    def log_exit():
        log.info("Exited")

    # setsid must be called as soon as possible, before config reading/compilation.
    # Otherwise instance may be killed with cqudp-session.
    if not args.no_detach and detect_cms_agent_type().is_bsconfig():
        log.info('Try to create new session and process group (do setsid) and redirect IO streams')
        detach_from_cms_agent()

    app = actionsutil.make_start_action(args.config, args.hq_url)
    try:
        app.run()
    finally:
        app.stop()


def add_parsers(subparsers):
    parser = subparsers.add_parser(name='start', description='Start instance')
    parser.set_defaults(handle=start_instance)
    parser.add_argument(
        "-c", "--config", metavar="CONFIGFILE", dest="config",
        help="config file", type=str, default='loop.conf',
    )
    parser.add_argument(
        "--no-detach", action='store_true',
        help="do not detach from CMS agent (do not do setsid)",
    )
    parser.add_argument(
        "--hq-url", metavar="HQ_URL", dest="hq_url",
        help="HQ URL", type=str, default=None,
    )

