# coding=utf-8
from __future__ import unicode_literals

import errno

import collections
import copy
import json
import logging
import os
import re
import shlex
import yaml
from ConfigParser import SafeConfigParser
from google.protobuf import json_format
from sepelib.util.fs import atomic_write, makedirs_ignore

from instancectl import cms
from instancectl import constants
from instancectl.cms import STATE_DIR, COMPILED_CONFIG_LOCK_PATH, COMPILED_CONFIG_PATH
from instancectl.config.errors import InstanceCtlConfigError
from instancectl.lib import envutil
from instancectl.lib.process.porto_container import PortoMode, VirtMode
from infra.nanny.instancectl.proto import instancectl_pb2
from instancectl.utils import FileLock
from . import defaults
from . import helpers

FORBIDDEN_SECTION_NAMES = ['instancectl', 'loop']


class RunConfig(object):
    """Contains run parameters for daemons.
    This class may be considered as a dict
    of dicts for sections with run parameters
    """

    LD_PRELOAD_ENV_VAR = 'LD_PRELOAD'
    MINIDUMPS_PATH_ENV_VAR = 'BREAKPAD_MINIDUMPS_PATH'
    DISABLE_INSTANCE_SIGNAL_HANDLER_ENV_VAR = 'NOSIGHANDLER'
    ANNOTATED_PORTS_OPTION = 'annotated_ports'

    PROPERTIES_OPTION = 'properties'
    HQ_POLL_ITAG = 'enable_hq_poll'
    HQ_REPORT_ITAG = 'enable_hq_report'

    def __init__(self, ctl_env):
        """
        :type ctl_env: instancectl.lib.envutil.InstanceCtlEnv
        """
        self.log = logging.getLogger(__name__ + '.' + self.__class__.__name__)
        self.total_defaults = defaults.total_defaults.copy()
        self._config = SafeConfigParser(self.total_defaults, dict_type=collections.OrderedDict)
        self.run_config_values = self.total_defaults.keys()
        self.run_config_values += defaults.mandatory_parameters
        self._ctl_env = ctl_env

    @staticmethod
    def _bool(value):
        false_values = [
            'no',
            'false',
            '0',
            ''
        ]
        return value.strip().lower() not in false_values

    @staticmethod
    def _parse_porto_mode(v):
        v = v.lower()
        if v == 'virt_mode_os':
            return PortoMode(enabled=True, isolate='true', virt_mode=VirtMode.OS)
        if v in ('true', 'full'):
            return PortoMode(enabled=True, isolate='false', virt_mode=VirtMode.APP)
        return PortoMode(enabled=False, isolate='false', virt_mode=VirtMode.APP)

    def load(self, config_file):
        self.log.debug("Loading config file: %s", config_file)

        if not self._config.read(config_file):
            raise InstanceCtlConfigError("No config file can be loaded: {}".format(config_file))

        for option, value in defaults.global_options.iteritems():
            self._config.set('DEFAULT', option, value)

        for s in FORBIDDEN_SECTION_NAMES:
            if self._config.has_section(s):
                raise InstanceCtlConfigError('Config section name cannot be equal to one of: {}, '
                                             'please choose other name'.format(FORBIDDEN_SECTION_NAMES))

        if not self._config.has_section('defaults'):
            self._config.add_section('defaults')

        # Обновляем DEFAULT опциями из defaults
        for option in self._config.options('defaults'):
            self._config.set(
                'DEFAULT',
                option,
                self._config.get('defaults', option, raw=1)
            )

    def save(self, destination_path):
        """
        Сериализует конфиг в yaml, записывает в :param destination_path: и возвращает конфиг в виде словаря

        :rtype: dict
        """
        jobs = collections.OrderedDict()

        config_globals = self.get_globals()

        for section in self.sections():
            jobs[section] = self.get_run_config(section, config_globals)

        if len(jobs) == 0:
            raise InstanceCtlConfigError('Cannot find any daemons in the conf file')

        result = {
            'jobs': jobs,
            'globals': config_globals,
        }

        dump = serialize_compiled_config(result)
        if self._ctl_env.skip_fdatasync_config:
            # fdatasync may be very slow in sandbox environment,
            # it cause test failures, so we disable it in test environment
            with open(destination_path, 'w') as fd:
                fd.write(dump)
        else:
            atomic_write(destination_path, dump)

        return result

    def _is_hq_poll_enabled(self, itags):
        return self.HQ_POLL_ITAG in itags

    def _is_hq_report_enabled(self, itags):
        return self.HQ_REPORT_ITAG in itags

    def get_globals(self):
        result = self._ctl_env.default_container_env.copy()
        result.update({
            'instance_dir': self._ctl_env.instance_dir,
            'instance_port': unicode(self._ctl_env.instance_port),
            'instance_name': self._ctl_env.instance_name,
            'node_name': self._ctl_env.node_name,
            'orthogonal_tags': self._ctl_env.orthogonal_tags,
            'auto_tags': self._ctl_env.auto_tags,
            'itags': self._ctl_env.instance_tags_string,
            'instance_host': self._ctl_env.default_container_env['BSCONFIG_IHOST'],
        })

        for option, default_value in defaults.global_options.iteritems():
            if self._config.has_option('defaults', option):
                result[option] = self._config.get('defaults', option)
            else:
                result[option] = default_value

        result['hq_min_report_delay'] = float(result['hq_min_report_delay'])
        result['hq_max_report_delay'] = float(result['hq_max_report_delay'])
        result['hq_max_report_delay_jitter'] = float(result['hq_max_report_delay_jitter'])
        result['hq_report_timeout'] = float(result['hq_report_timeout'])
        result['hq_report'] = self._is_hq_report_enabled(result['itags'])
        result['hq_poll'] = self._is_hq_poll_enabled(result['itags'])

        result['status_update_max_restart_period'] = float(result['status_update_max_restart_period'])
        result['status_update_min_restart_period'] = float(result['status_update_min_restart_period'])
        result['status_update_restart_period_backoff'] = float(result['status_update_restart_period_backoff'])

        result['its_max_poll_timeout'] = float(result['its_max_poll_timeout'])
        result['its_connection_timeout'] = float(result['its_connection_timeout'])
        result['its_max_timeout_jitter'] = float(result['its_max_timeout_jitter'])
        if result['its_force_poll_timeout']:
            result['its_force_poll_timeout'] = float(result['its_force_poll_timeout'])
        else:
            result['its_force_poll_timeout'] = None
        result['action_stop_timeout'] = float(result['action_stop_timeout'])

        result['status_check_tries'] = int(result['status_check_tries'])
        result['status_check_min_delay'] = float(result['status_check_min_delay'])
        result['status_check_backoff'] = float(result['status_check_backoff'])
        result['status_check_max_delay'] = float(result['status_check_max_delay'])
        result['status_check_req_timeout'] = float(result['status_check_req_timeout'])
        result['liveness_criterion'] = helpers.make_liveness_criterion_from_config(result, self.sections())

        if cms.detect_cms_agent_type().is_iss():
            dump_json_content = cms.load_dump_json()
            dump_json_props = dump_json_content.get('properties', {})
            result['annotated_ports'] = self.get_annotated_ports(dump_json_props)
        return result

    def sections(self):
        config_sections = self._config.sections()
        if 'defaults' in config_sections:
            config_sections.remove('defaults')
        return config_sections

    def get(self, section, attribute):
        return self._config.get(section, attribute)

    def get_conf_local_lines(self, section, config_file_path=None):
        """It's a good thing to keep this function thread-safe."""
        conf_local_lines = []
        if config_file_path is None:
            config_file_path = self._config.get(section, 'Conf.local')
        if os.path.exists(config_file_path):
            try:
                with open(config_file_path) as fd:
                    conf_local_lines = fd.read().splitlines()
            except IOError as e:
                if e.errno != errno.ENOENT:
                    raise
        return conf_local_lines

    def match_run_parameters(self, match_regex, lines):
        """Given a match regex and possible lines with run parameters extract
        this parameters. Usually "lines" contains Conf.local lines and splitted
        BSCONFIG_ITAGS environment variable value.

        Example:
            Input: '^ENV_(.*)$' regex and ['ctype_base', 'ENV_NCPU=16']
            Output: ('NCPU', '16')


        Also note, that it's a good thing to keep this function thread-safe
        """
        for line in lines:
            line = line.strip()
            self.log.debug('Matching "%s" with "%s"', line, match_regex)
            match = re.match(match_regex, line)
            if not match:
                continue
            parameter_line = match.group(1)
            if '=' not in parameter_line:
                self.log.error(
                    'No "=" symbol found in result of '
                    'matching "%s" with "%s".',
                    line, match_regex
                )
                continue
            variable, value = parameter_line.split('=', 1)
            if not (variable and value):
                self.log.error(
                    'Got zero variable or value while matching line %s.',
                    line
                )
                continue
            yield (variable, value)

    def _handle_matches(self, conf_match, env_match, extra_getconf_variables, opt_match, run_config, section):
        """
        Ищем опции по регуляркам conf_match, env_match, opt_match
        и обновляем ими extra_getconf_variables, run_config
        """
        #
        # BSCONFIG_ITAGS should have priority over Conf.local lines.
        # Down below it's achieved parsing Conf.local first and
        # (possibly) overwriting results later, while parsing ITAGS
        #
        extraconf_lines = self.get_conf_local_lines(section)
        extraconf_lines += shlex.split(self._ctl_env.instance_tags_string)

        if env_match:
            for variable, value in self.match_run_parameters(
                    env_match, extraconf_lines):
                self.log.debug('value: %s', value)
                value = value % self._ctl_env.default_container_env
                self.log.debug(
                    'Updating environment with %s=%s.',
                    variable,
                    value
                )
                run_config['environment'][variable] = value
                run_config['environment']['LOOP_' + variable] = value
                extra_getconf_variables[variable] = value

        if opt_match:
            for variable, value in self.match_run_parameters(
                    opt_match, extraconf_lines
            ):
                self.log.debug(
                    'Updating options with %s=%s.',
                    variable,
                    value
                )
                extra_getconf_variables[variable] = value
                run_config['extra_run_arguments'] += ['-V', '%s=%s' % (variable, value)]

        if conf_match:
            for variable, value in self.match_run_parameters(
                    conf_match, extraconf_lines):
                self.log.debug(
                    'Updating options with %s=%s.',
                    variable,
                    value
                )
                extra_getconf_variables[variable] = value
                run_config['environment']['LOOP_' + variable] = value
                run_config['extra_run_arguments'] += ['-V', '%s=%s' % (variable, value)]

    def get_annotated_ports(self, iss_properties):
        """
        Составляет словарь аннотированных портов, получаемых из dump.json:
        :param iss_properties: кусок dump.json от звездолёта
        :type iss_properties: dict
        :rtype: dict[str | unicode, int]
        """
        ports_json = iss_properties.get(self.ANNOTATED_PORTS_OPTION)
        if ports_json is None:
            return {}
        return json.loads(ports_json)

    def get_run_config(self, section, config_globals=None):
        """It's a good thing to keep this function thread-safe."""
        #
        # We should set up extra_getconf_variables.
        # The goal is to allow BSCONFIG_* environment
        # variables, Conf.local lines and BSCONFIG_ITAGS
        # to be available in config
        #

        # FIXME Filter only BSCONFIG_* environment?
        render_vars = {}
        for k, v in self._ctl_env.default_container_env.iteritems():
            render_vars[k] = v

        instance_port = self._ctl_env.instance_port
        if instance_port is not None:
            v = envutil.make_gencfg_ports_render_vars(int(instance_port))
            render_vars.update(v)

        if cms.detect_cms_agent_type().is_iss():
            ports = config_globals.get('annotated_ports')
            if ports is not None:
                # Преобразуем порты к виду:
                # {
                #     "annotated_ports_main": 8080,
                #     "annotated_ports_extra": 8081,
                # }
                annotated_ports = {'annotated_ports_{}'.format(n): str(v) for n, v in ports.iteritems()}
                render_vars.update(annotated_ports)

        render_vars.update(config_globals['orthogonal_tags'])

        render_vars['section'] = section
        run_config = {
            'environment': self._ctl_env.default_container_env.copy(),
            'extra_run_arguments': [],
            'globals': config_globals or {},
        }

        run_config['environment']['BSCONFIG_LOOPSECTION'] = section

        env_match = self._config.get(section, 'env_match')
        opt_match = self._config.get(section, 'opt_match')
        conf_match = self._config.get(section, 'conf_match')

        if env_match or opt_match or conf_match:
            self._handle_matches(
                conf_match,
                env_match,
                render_vars,
                opt_match,
                run_config,
                section
            )
        #
        # Now time to populate run_config.
        # Value resolution order does not matter, I think.
        # Also note, that in SafeConfigParser.get method vars dictionary
        # has priority over section and DEFAULT values.
        #
        for item in self.run_config_values:
            run_config[item] = self._config.get(
                section, item, raw=False, vars=render_vars
            )

        # Special type conversions
        for var in [
            'coredump_probability', 'delay', 'backoff',
            'max_delay', 'max_jitter', 'kill_timeout', 'terminate_timeout',
            'coredump_probability', 'minidumps_check_timeout',
            'minidumps_timeout_before_sending', 'minidumps_clean_timeout',
            'coredumps_send_overall_timeout', 'coredumps_gdb_stackwalk_timeout'
        ]:
            try:
                val = float(run_config[var])
            except (TypeError, ValueError):
                self.log.critical('Invalid value %s for variable %s', run_config[var], var)
                raise ValueError('Invalid value %s for variable %s' % (run_config[var], var))
            run_config[var] = val

        run_config['coredumps_count_limit'] = int(run_config['coredumps_count_limit'])
        run_config['always_coredump'] = self._bool(run_config['always_coredump'])
        porto_mode = self._parse_porto_mode(run_config['use_porto'])
        if porto_mode.enabled and not self._is_porto_enabled(config_globals['itags']):
            raise InstanceCtlConfigError('Option "use_porto" is set but porto is disabled: '
                                         'deploy engine is not ISS and there is no "portopowered" tag')
        run_config['use_porto'] = porto_mode

        self.add_minidumps_environment(section, run_config, run_config['environment'])

        self.log.debug('section: %r', section)
        eval_variables = self.add_eval_variables(section, render_vars)
        run_config.update(eval_variables)
        run_config['status_check'] = helpers.make_container_liveness_check(run_config, config_globals)
        run_config['legacy_process_liveness_check'] = helpers.is_legacy_process_liveness_check_required(run_config,
                                                                                                        config_globals)
        return run_config

    def add_eval_variables(self, section, extra_variables):
        eval_vars = {}
        self.log.debug('args in this section: %s', self._config.options(section))
        for opt in self._config.options(section):
            if opt.startswith('eval_'):
                eval_vars[opt] = self._config.get(
                    section, opt, raw=False, vars=extra_variables
                )
        self.log.debug('eval_vars: %r', eval_vars)
        return eval_vars

    def add_minidumps_environment(self, section, run_config, environment):
        """
        Добавляем переменные окружения, необходимые для работы Breakpad:
            1. LD_PRELOAD для подгрузки библиотеки для откладывания minidump'ов
            2. BREAKPAD_MINIDUMPS_PATH для указания директории, куда откладывать
            minidump'ы

        :type run_config: dict
        :type environment: dict
        """
        if not run_config['minidumps_push']:
            return
        if run_config['coredumps_format'] != 'minidump':
            return
        self.log.info(
            'Patching %s env variable of section %s, adding %s to it',
            self.LD_PRELOAD_ENV_VAR, section, run_config['minidumps_library']
        )
        old_ld_preload = environment.get(self.LD_PRELOAD_ENV_VAR)
        if not old_ld_preload:
            ld_preload = run_config['minidumps_library']
        else:
            self.log.info('Old %s of %s: %s', self.LD_PRELOAD_ENV_VAR, section, old_ld_preload)
            ld_preload = ' '.join((old_ld_preload, run_config['minidumps_library']))
        self.log.info('New %s of %s: %s', self.LD_PRELOAD_ENV_VAR, section, ld_preload)
        environment[self.LD_PRELOAD_ENV_VAR] = ld_preload

        if not run_config['minidumps_path']:
            run_config['minidumps_path'] = './minidumps/{}/'.format(section)
        self.log.info(
            'Patching %s env variable of %s, setting it to %s',
            self.MINIDUMPS_PATH_ENV_VAR, section, run_config['minidumps_path']
        )
        old_minidumps_path = environment.get(self.MINIDUMPS_PATH_ENV_VAR)
        if old_minidumps_path:
            self.log.info('Old %s of %s: %s', self.MINIDUMPS_PATH_ENV_VAR, section, old_minidumps_path)
        environment[self.MINIDUMPS_PATH_ENV_VAR] = run_config['minidumps_path']

        self.log.info('Setting {} env variable to disable signal handling by instance'.format(
            self.DISABLE_INSTANCE_SIGNAL_HANDLER_ENV_VAR
        ))
        environment[self.DISABLE_INSTANCE_SIGNAL_HANDLER_ENV_VAR] = '1'

    @staticmethod
    def _is_porto_enabled(tags):
        """
        Проверяем, разрешено ли запускать бинарники в porto-контейнерах, это можно в любом из двух случаев:
          * Мы запущены под ISS
          * Мы запущены под bsconfig с тэгом portopowered

        :type tags: list[unicode]
        :rtype: bool
        """
        return cms.detect_cms_agent_type().is_iss() or constants.PORTOPOWERED_ITAG in tags


def compile_instance_config(config_file, ctl_env):
    """
    Компилирует конфиг инстанса из :param config_file:, Conf.local и переменных окружения BSCONFIG_*,
    возвращает полученный конфиг

    :type config_file: str
    :type ctl_env: instancectl.lib.envutil.InstanceCtlEnv
    :rtype dict:
    """
    makedirs_ignore(STATE_DIR)

    with FileLock(COMPILED_CONFIG_LOCK_PATH):
        config = RunConfig(ctl_env)
        config.load(config_file)
        return config.save(COMPILED_CONFIG_PATH)


def get_instance_config():
    """
    Получает скомпилированный конфиг инстанса
    """
    try:
        with open(COMPILED_CONFIG_PATH) as fd:
            compiled_config = read_compiled_config(fd)
    except Exception as e:
        raise InstanceCtlConfigError('Cannot load compiled config: {}'.format(e))

    # Конфиг может оказаться пустым: SWAT-2062
    if not compiled_config:
        raise InstanceCtlConfigError('Compiled config is empty, need to recompile it')

    return compiled_config


def get_or_compile_instance_config(config_file, ctl_env):
    """
    Возвращает скомпилированных конфиг инстанса, если такой уже есть, или компилирует его

    :type config_file: str | unicode
    :type ctl_env: instancectl.lib.envutil.InstanceCtlEnv
    """
    try:
        return get_instance_config()
    except InstanceCtlConfigError:
        return compile_instance_config(config_file, ctl_env)


def serialize_compiled_config(config):
    result = copy.deepcopy(config)
    for j in result['jobs'].itervalues():
        j['status_check'] = json_format.MessageToDict(j['status_check'])
    g = result['globals']
    g['liveness_criterion'] = json_format.MessageToDict(g['liveness_criterion'])
    return yaml.dump(result)


def read_compiled_config(fd):
    loaded = ordered_load(fd)
    for j in loaded['jobs'].itervalues():
        c = instancectl_pb2.ContainerLivenessCheck()
        json_format.ParseDict(j['status_check'], c)
        j['status_check'] = c
    g = loaded['globals']
    c = instancectl_pb2.InstanceLivenessCriterion()
    json_format.ParseDict(g['liveness_criterion'], c)
    g['liveness_criterion'] = c
    return loaded


def ordered_load(stream, Loader=yaml.Loader, object_pairs_hook=collections.OrderedDict):
    class OrderedLoader(Loader):
        pass

    def construct_mapping(loader, node):
        loader.flatten_mapping(node)
        return object_pairs_hook(loader.construct_pairs(node))

    OrderedLoader.add_constructor(
        yaml.resolver.BaseResolver.DEFAULT_MAPPING_TAG,
        construct_mapping)
    return yaml.load(stream, OrderedLoader)
