from __future__ import unicode_literals

from instancectl.lib import pbutil
from infra.nanny.instancectl.proto import instancectl_pb2
from instancectl.utils import parse_config_int
from . import errors


def liveness_criterion_type_to_value(s):
    try:
        return pbutil.enum_name_to_value(instancectl_pb2.InstanceLivenessCriterion.DESCRIPTOR, s.upper())
    except ValueError:
        raise errors.StatusCheckConfigError('Unknown liveness_criterion_type: %s', s)


def make_liveness_criterion_from_config(config, sections):
    """
    :type config: dict
    :rtype: instancectl_pb2.InstanceLivenessCheck
    """
    result = instancectl_pb2.InstanceLivenessCriterion()
    result.type = liveness_criterion_type_to_value(config['liveness_criterion_type'])
    if result.type == instancectl_pb2.InstanceLivenessCriterion.LIST:
        if not config['liveness_criterion_list']:
            raise errors.StatusCheckConfigError('liveness_criterion_type = list but liveness_criterion_list is empty')
        containers = config['liveness_criterion_list'].split(',')
        for c in containers:
            c = c.strip()
            if c not in sections:
                raise errors.StatusCheckConfigError('Unknown section name "{}" in liveness_criterion_list'.format(c))
            item = result.container_list.list.add()
            item.container_id = c
    return result


def status_check_type_to_value(s):
    try:
        return pbutil.enum_name_to_value(instancectl_pb2.AdditionalContainerLivenessCheck.DESCRIPTOR, s.upper())
    except ValueError:
        raise errors.StatusCheckConfigError('Unknown status_check_type: {}'.format(s))


def list_optional_checks(config):
    """
    :type config: dict
    :rtype: list[unicode]
    """
    check_type = config.get('status_check_type')
    if not check_type:
        return []
    if check_type != 'list':
        return [check_type.strip()]
    if not config['status_check_list']:
        raise errors.StatusCheckConfigError('status_check_type = list selected but status_check_list is empty')
    return [c.strip() for c in config['status_check_list'].split(',')]


def fill_optional_status_checks(checks, config, result):
    for check in checks:
        if check == 'process_liveness':
            continue
        check_pb = result.optional_checks.add()
        check_pb.type = status_check_type_to_value(check)
        if check_pb.type == instancectl_pb2.AdditionalContainerLivenessCheck.SCRIPT:
            if not config['status_script']:
                raise errors.StatusCheckConfigError('Status check type "script" selected but '
                                                    '"status_script" is not given')
            check_pb.script.script = config['status_script']
            check_pb.script.timeout_seconds = parse_config_int(config, 'status_check_timeout')


def _make_container_liveness_check(config):
    """
    :type config: dict
    :rtype: infra.nanny.instancectl.proto.instancectl_pb2.ContainerLivenessCheck
    """
    result = instancectl_pb2.ContainerLivenessCheck()
    result.process_liveness.required_lifetime_seconds = parse_config_int(config, 'status_check_required_lifetime')
    checks = list_optional_checks(config)
    if config['status_script'] and 'script' not in checks:
        raise errors.InstanceCtlConfigError('status_script is given but status_check_type is not "script"')
    fill_optional_status_checks(checks, config, result)
    return result


def _make_legacy_container_liveness_check(config):
    """
    :rtype: infra.nanny.instancectl.proto.instancectl_pb2.ContainerReadinessCheck
    """
    result = instancectl_pb2.ContainerLivenessCheck()
    result.process_liveness.required_lifetime_seconds = parse_config_int(config, 'status_check_required_lifetime')
    checks = list_optional_checks(config)

    # To preserve backward compatibility we must make status script check even
    # if check_type is not 'script' and no 'script' given in 'status_check_list'
    # but 'status_script' is given
    if config['status_script'] and 'script' not in checks:
        checks.append('script')

    fill_optional_status_checks(checks, config, result)
    return result


def make_container_liveness_check(config, global_config):
    """
    :type config: dict
    :type global_config: dict
    :rtype: infra.nanny.instancectl.proto.instancectl_pb2.ContainerReadinessCheck
    """
    if global_config['use_true_liveness']:
        return _make_container_liveness_check(config)
    else:
        return _make_legacy_container_liveness_check(config)


def is_legacy_process_liveness_check_required(config, global_config):
    """
    :type config: dict
    :type global_config: dict
    :rtype: bool
    """
    if global_config['use_true_liveness']:
        return False
    if config['status_check_type'] == 'process_liveness':
        return True
    if config['status_check_type'] == 'list' and 'process_liveness' in list_optional_checks(config):
        return True
    return False
