from __future__ import unicode_literals

import logging
import os
import re

from instancectl import utils


class ICoreDumpProcessor(object):

    def is_appropriate_dump(self, file_name):
        """
        :type file_name: unicode
        :rtype: bool
        """
        raise NotImplementedError

    def remove_dump(self, file_name):
        """
        :type file_name: unicdoe
        """
        raise NotImplementedError


class CoreDumpProcessor(ICoreDumpProcessor):

    SENT_CORE_FILE_NAME_SUFFIX = '.sent'

    def __init__(self, binary_name, core_pattern=None):
        self._binary_name = os.path.basename(binary_name)
        self._core_pattern = core_pattern
        self._log = logging.getLogger('coredump-proc')

    def is_appropriate_dump(self, file_name):
        self._log.info('Checking core file "%s" by pattern "%s"', file_name, self._core_pattern)
        base_name = os.path.basename(file_name)
        # We use special suffix to mark coredump as sent to avoid
        # multiple requests for the dump
        if base_name.endswith(self.SENT_CORE_FILE_NAME_SUFFIX):
            return False

        if self._core_pattern:
            return re.match(self._core_pattern, base_name) is not None

        return base_name.startswith(self._binary_name)

    def remove_dump(self, file_name):
        os.rename(file_name, '{}{}'.format(file_name, self.SENT_CORE_FILE_NAME_SUFFIX))


class MiniDumpProcessor(ICoreDumpProcessor):

    def __init__(self, _, __):
        self._log = logging.getLogger('minidump-proc')

    def is_appropriate_dump(self, file_name):
        self._log.info('Core pattern is not set, assuming file "%s" as appropriate dump', file_name)
        return True

    def remove_dump(self, file_name):
        utils.remove_file_if_exists(file_name)
