# -*- coding: utf-8 -*-
"""
Логика сбора минидампов и их отправки в агрегатор.
"""
import os
import sys
import time
import errno
import logging
import tempfile
import subprocess

import gevent

from instancectl.lib import minidumputil

from enum import Enum
from sepelib.gevent import GreenThread
from sepelib.subprocess.util import terminate
from sepelib.util import fs
from minidump2core import minidump_text_to_core

from instancectl.lib.coreaggrutil import is_new_cores_aggregator, ParseResult
from instancectl.lib.procutil import ExecuteHelper
from instancectl.utils import file_write_via_renaming, ensure_path
from . import processors


class MinidumpSenderError(Exception):
    pass


class SymbolsError(MinidumpSenderError):
    pass


class MinidumpStackwalkError(MinidumpSenderError):
    pass


class MinidumpToGdbTracesError(MinidumpSenderError):
    pass


class AggregatorType(Enum):
    SOCORRO = 'socorro'
    SAAS_AGGREGATOR = 'saas_aggregator'


class Backend(Enum):
    MINIDUMP = 'minidump'
    COREDUMP = 'coredump'


DUMP_FILE_NAME_FILTERS = {
    Backend.MINIDUMP: processors.MiniDumpProcessor,
    Backend.COREDUMP: processors.CoreDumpProcessor,
}

DUMPS_DIR_MAPPING = {
    Backend.MINIDUMP: 'minidumps_path',
    Backend.COREDUMP: 'coredumps_dir',
}


class MinidumpSender(GreenThread):
    """
    Отправляет отложенные minidump'ы агрегатору с помощью POST-запросов по заданному url'у.

    Поскольку мы не патчим сокеты, долгие POST-запросы блокируют весь луп-скрипт,
    поэтому непосредственную отправку мы делаем в отдельном процессе.

    :type _processor: instancectl.coredumps.processors.ICoreDumpProcessor
    """

    MINIDUMP_CHECK_TIME_FILE = 'state/minidump.mtime'
    TEMP_DIR = 'state/tmp/'
    SOCKET_TIMEOUT = 5.0
    MINIDUMP_STACKWALK_TIMEOUT = 30.0
    MINIDUMP_STACKWALK_TERMINATE_TIMEOUT = 5.0
    PROCESS_CHECK_TIMEOUT = 0.5
    SYMBOLS_DIR_NAME = './debug_symbols'

    def __init__(self, section, run_config, aggregator_type, coredump_format, limits):
        """
        :type section: unicode
        :type run_config: dict
        :type aggregator_type: AggregatorType
        :type coredump_format: Backend
        """
        super(MinidumpSender, self).__init__()
        self.log = logging.getLogger('{}.{}'.format(__name__, self.__class__.__name__))
        self._enabled = run_config['minidumps_push']
        self._section = section
        self._url = run_config['minidumps_aggregator_url']
        self._minidumps_path = run_config[DUMPS_DIR_MAPPING[coredump_format]]
        self._service = run_config['minidumps_service']
        self._timeout_before_sending = run_config['minidumps_timeout_before_sending']
        self._check_timeout = run_config['minidumps_check_timeout']
        self._clean_timeout = run_config['minidumps_clean_timeout']
        self._instance_name = run_config['environment']['BSCONFIG_INAME']
        self._minidumps_stackwalk = run_config['minidumps_stackwalk']
        self._binary = run_config['rename_binary'] or run_config['binary']
        self._core_pattern = run_config['core_pattern']
        self._symbols_file = run_config['minidumps_symbols']
        self._gdb_path = run_config['coredumps_gdb_path']
        self._symbols_dir = None
        self._aggregator_type = aggregator_type
        self._backend = coredump_format
        self._ctype = self._get_ctype_from_itags(run_config['globals']['itags'])
        self._prj = self._get_prj_from_itags(run_config['globals']['itags'])
        self._instancectl = os.path.join('.', os.path.basename(sys.argv[0]))
        self._limits = limits
        self._processor = DUMP_FILE_NAME_FILTERS[coredump_format](self._binary, run_config['core_pattern'])
        self._overall_timeout = run_config['coredumps_send_overall_timeout']
        self._gdb_stackwalk_timeout = run_config['coredumps_gdb_stackwalk_timeout']

        # плохо запоминать целиком run_config, но оставляем это
        # для использования в ExecuteHelper'е
        self._run_config = run_config

    def _get_symbols_path(self):
        """
        Складываем символы бинарника в структуру директорий,
        понятную breakpad'у.
        """
        if self._symbols_dir is None:
            try:
                with open(self._symbols_file) as fd:
                    header = fd.readline()

            except Exception as err:
                self.log.exception('Cannot read debug symbols')
                raise SymbolsError('Cannot read debug symbols: {}'.format(err))

            try:
                # заголовок минидампа обычно имеет вид:
                # MODULE Linux x86_64 6EDC6ACDB282125843FD59DA9C81BD830 httpsearch
                # вырезаем из него идентификатор символов
                binary_id = header.split()[3]
            except Exception as err:
                self.log.exception('Cannot parse debug symbols')
                raise SymbolsError('Cannot parse debug symbols: {}'.format(err))

            # minidump_stackwalk ожидает такую структуру директории с символами:
            # symbols_dir/<binary_name>/6EDC6ACDB282125843FD59DA9C81BD830/<binary_name>.sym
            #
            # Для парсинга минидампа утилитой minidump_stackwalk необходимо совпадение
            # имени <binary_name> с именем бинарника, осевшем в минидампе. В зависимости от
            # платформы в минидампе может осесть как реальное имя бинарника, так и имя симлинка
            # на него. Поэтому мы создаём симлинки символов, соответствующие обоим этим вариантам

            binary_names = {
                os.path.basename(self._binary),
                os.path.basename(os.path.realpath(self._binary)),
            }

            for name in binary_names:
                dir_for_symbols = os.path.join(self.SYMBOLS_DIR_NAME, name, binary_id)
                symbols_path = os.path.join(dir_for_symbols, name + '.sym')
                try:
                    ensure_path(dir_for_symbols)
                    os.link(self._symbols_file, symbols_path)
                except OSError as err:
                    if err.errno != errno.EEXIST:
                        self.log.error('Cannot link debug symbols', exc_info=True)
                        raise SymbolsError('Cannot link debug symbols: {}'.format(err))
                except Exception as err:
                    self.log.error('Cannot link debug symbols', exc_info=True)
                    raise SymbolsError('Cannot link debug symbols: {}'.format(err))

            self._symbols_dir = self.SYMBOLS_DIR_NAME

        return self._symbols_dir

    def _get_ctype_from_itags(self, itags):
        """
        Извлекает ctype из itag'ов

        :param str itags: itag'и разделенные пробелом
        :return: ctype или None, если не удалось его определить
        """
        for tag in itags.split():
            if tag.startswith(minidumputil.CTYPE_ITAG_PREFIX):
                return tag[len(minidumputil.CTYPE_ITAG_PREFIX):]

    def _get_prj_from_itags(self, itags):
        """
        Извлекает prj из itag'ов

        :param str itags: itag'и разделенные пробелом
        :return: prj или None, если не удалось его определить
        """
        for tag in itags.split():
            if tag.startswith(minidumputil.PRJ_ITAG_PREFIX):
                return tag[len(minidumputil.PRJ_ITAG_PREFIX):]

    def _get_last_check_time(self):
        """
        Определяет минимальное время создания минидампа,
        который нужно отправить в агрегатор.

          * пытается извлечь время из файла
          * если не удалось, возвращает текущее время
        """
        try:
            with open(self.MINIDUMP_CHECK_TIME_FILE) as fd:
                return float(fd.read())
        except Exception as err:
            self.log.info('Cannot load last minidump check time %s, will upload only minidumps '
                          'created later than now: %s', self.MINIDUMP_CHECK_TIME_FILE, err)
            return time.time()

    def _run_subprocess(self, command, stdout=None, stderr=None, timeout=None, exception=MinidumpSenderError):
        if timeout is None:
            timeout = self._overall_timeout
        execute_helper = ExecuteHelper(self._limits)

        with open(os.devnull, 'a') as devnull:
            process = subprocess.Popen(
                command,
                preexec_fn=execute_helper,
                stdout=stdout or devnull,
                stderr=stderr or devnull,
            )

            deadline = time.time() + timeout
            rc = process.poll()
            while rc is None and time.time() < deadline:
                gevent.sleep(self.PROCESS_CHECK_TIMEOUT)
                rc = process.poll()

        if rc is None:
            self.log.error('Command %s timeout: %s seconds, killing it', command, timeout)
            try:
                terminate(process)
            except Exception as err:
                self.log.error('Cannot kill timed out process', exc_info=True)
                raise exception('Cannot kill timed out process: {}'.format(err))
            else:
                raise exception('Command {} timed out: {} seconds'.format(command, timeout))

        if rc != 0:
            self.log.error('Command %s fail: exit code %s; will not send minidump to aggregator', command, rc)
            raise exception('Command {} fail: exit code {}'.format(command, rc))

    def _run_send_minidump_command(self, filename):
        self._run_subprocess(
            [self._instancectl, 'send_minidump', self._section, filename],
            timeout=self._overall_timeout,
        )

    def _parse_minidump(self, filename):
        """
        Распарсить минидамп с помощью утилиты minidump_stackwalk
        """
        ensure_path(self.TEMP_DIR)
        with tempfile.TemporaryFile(dir=self.TEMP_DIR) as stackwalk_output:
            # здесь нельзя использовать stdout=PIPE из-за слишком
            # большого вывода minidump_stackwalk, который блочит процесс
            self._run_subprocess(
                [self._minidumps_stackwalk, '-m', filename, self._get_symbols_path()],
                stdout=stackwalk_output,
                timeout=self.MINIDUMP_STACKWALK_TIMEOUT,
                exception=MinidumpStackwalkError,
            )

            try:
                stackwalk_output.seek(0)
                return stackwalk_output.read()
            except Exception as err:
                self.log.error('Cannot read minidump_stackwalk results', exc_info=True)
                raise MinidumpStackwalkError('Cannot read minidump_stackwalk results: {}'.format(err))

    def _send_minidump_to_saas(self, filename):
        """
        Отправка минидампа/корки в SaaS агрегатор
        @param filename: имя минидампа или корки
        """
        parse_result = ParseResult.OK
        parsed_traces = ""
        parse_error = ""
        if self._backend == Backend.MINIDUMP:
            parsed_minidump = self._parse_minidump(filename)
            try:
                parsed_traces = minidump_text_to_core(parsed_minidump)
            except Exception as err:
                msg = 'Cannot parse minidump_stackwalk results via minidump2core'
                self.log.error(msg, exc_info=True)
                parse_result = ParseResult.FAILED
                if not is_new_cores_aggregator(self._url):
                    raise MinidumpToGdbTracesError(
                        'Cannot parse minidump_stackwalk results via minidump2core: {}'.format(err)
                    )
                parse_error = msg
        elif self._backend == Backend.COREDUMP:
            try:
                parsed_traces = minidumputil.extract_coredump_traces(
                    file_name=filename,
                    binary=self._binary,
                    gdb_path=self._gdb_path,
                    gdb_timeout=self._gdb_stackwalk_timeout,
                    limits=self._limits
                )
            except Exception as err:
                msg = 'Cannot parse coredump results via gdb: {}'.format(err)
                self.log.error(msg, exc_info=True)
                parse_result = ParseResult.FAILED
                if not is_new_cores_aggregator(self._url):
                    raise MinidumpToGdbTracesError(msg)
                parse_error = msg
        else:
            raise MinidumpSenderError('Unknown coredump format: {}'.format(self._backend))
        minidumputil.send_minidump_to_saas(
            file_name=filename,
            parsed_traces=parsed_traces,
            url=self._url,
            service=self._service,
            instance_name=self._instance_name,
            ctype=self._ctype,
            parse_error=parse_error,
            parse_result=parse_result,
        )

    def send_minidump(self, filename):
        """
        Отправка минидампа в агрегатор
        """
        if self._aggregator_type == AggregatorType.SAAS_AGGREGATOR:
            self._send_minidump_to_saas(filename)
        else:
            raise MinidumpSenderError('Unsupported aggregator type: {}'.format(self._aggregator_type))

    def run(self):
        """
        Основной цикл, отслеживающий появление новых минидампов или корок
        """
        if not self._enabled:
            return

        if self._backend == Backend.MINIDUMP:
            try:
                fs.makedirs_ignore(self._minidumps_path)
            except Exception:
                self.log.warning('Cannot create dir for minidumps, minidumps saving may not work', exc_info=True)

        last_check_time = self._get_last_check_time()

        while True:
            self.log.info('Collecting minidumps/cores at "%s"...', self._minidumps_path)
            # breakpad/ядро производят запись не атомарно, поэтому отправляем
            # только те файлы, запись в которые закончена достаточно давно
            check_time = time.time() - self._timeout_before_sending

            try:
                file_names = os.listdir(self._minidumps_path)
            except Exception as err:
                self.log.info('Cannot get list of files in dumps dir %s: %s', self._minidumps_path, err)
            else:
                for file_name in file_names:
                    file_name = os.path.join(self._minidumps_path, file_name)
                    if not self._processor.is_appropriate_dump(file_name):
                        continue
                    try:
                        mtime = os.path.getmtime(file_name)
                    except Exception as err:
                        self.log.info('Cannot get mtime of file %s: %s', file_name, err)
                        continue

                    if mtime < check_time:

                        # отправляем только файлы, появившиеся после предыдущей проверки
                        if mtime > last_check_time:
                            self.log.info('Sending dump %s to aggregator', file_name)
                            try:
                                self._run_send_minidump_command(file_name)
                            except Exception as err:
                                self.log.info('Cannot send dump %s to aggregator: %s', file_name, err)

                        if self._clean_timeout and mtime < time.time() - self._clean_timeout:
                            # удаляем файл независимо от того, был ли он отправлен в агрегатор
                            self.log.debug('Removing dump %s', file_name)
                            try:
                                self._processor.remove_dump(file_name)
                            except Exception as err:
                                self.log.info('Cannot remove dump %s: %s', file_name, err)

            try:
                file_write_via_renaming(self.MINIDUMP_CHECK_TIME_FILE, str(check_time), self.TEMP_DIR)
            except Exception as err:
                self.log.info('Cannot write last dumps check time to %s: %s', self.MINIDUMP_CHECK_TIME_FILE, err)

            last_check_time = check_time

            gevent.sleep(self._check_timeout)


# FIXME: temporary hack, remove it
class EmptyMinidumpSender(object):

    def start(self):
        return
