from __future__ import unicode_literals
import collections

from instancectl.lib import envutil
from instancectl.hq import instance_client


def get_cluster_name(c):
    name = c.meta.name.rsplit('_', 1)[0]
    return name.upper()


class HqInstancesGetter(object):
    def __init__(self, federated_client, env):
        self.federated_client = federated_client
        self.env = env
        self.query_pool = {}
        self._clusters = None

    @property
    def clusters(self):
        if self._clusters is None:
            self._clusters = collections.OrderedDict()
            for c in self.federated_client.find_clusters():
                self._clusters[get_cluster_name(c)] = c
        return self._clusters

    def _hq_cluster_instances(self, service_id, cluster_name):
        c = self.clusters.get(cluster_name)
        if not c:
            return

        hq_url = c.spec.endpoint.url
        hq_client = instance_client.HqInstanceClient(hq_url)
        instances = hq_client.find_instances(service_id)
        for instance in instances:
            yield instance

    def get_cluster_instances(self, service_id, cluster_name):
        key = (cluster_name, service_id)
        if key not in self.query_pool:
            instances = list(self._hq_cluster_instances(service_id, cluster_name))
            self.query_pool[key] = instances
        return self.query_pool[key]

    def get_instances(self, service_id, cluster_names=None):
        if cluster_names is None:
            cluster_names = self.clusters.keys()

        rv = []
        for name in cluster_names:
            name = name.upper()
            rv.extend(self.get_cluster_instances(service_id, name))
        return rv

    def get_instances_current_cluster(self, service_id):
        loc = envutil.extract_location_from_orthogonal_tags(self.env.orthogonal_tags)
        if loc == envutil.Location.UNKNOWN:
            raise ValueError("Could not get current cluster name")
        return self.get_instances(service_id, cluster_names=[loc.name])
