from __future__ import unicode_literals

import nanny_rpc_client
import requests
from nanny_rpc_client import exceptions
from sepelib.util.retry import Retry, RetrySleeper

from yp_proto.yp.client.hq.proto import hq_pb2
from infra.nanny2.stubs.python import hq_stub
from instancectl.lib import envutil


class HqInstanceClient(object):

    _DEFAULT_REQ_TIMEOUT = 5.0

    def __init__(self, url, req_timeout=None):
        """
        :type url: unicode
        """
        self.req_timeout = req_timeout or self._DEFAULT_REQ_TIMEOUT
        instance_url = '{}/rpc/instances/'.format(url.rstrip('/'))
        hq_rpc = nanny_rpc_client.RequestsRpcClient(rpc_url=instance_url, request_timeout=self.req_timeout)
        self._hq_stub = hq_stub.InstanceServiceStub(hq_rpc)
        self._retry = Retry(retry_sleeper=RetrySleeper(max_tries=3, delay=1, backoff=3),
                            retry_exceptions=(exceptions.RpcError, requests.RequestException))
        self._last_reported_conf_id = None

    @staticmethod
    def _get_revision_id():
        """
        :rtype: unicode
        """
        return envutil.get_configuration_id()

    def get_instance_revision(self, instance_id):
        """
        :type instance_id: unicode
        :rtype: types_pb2.InstanceRevision
        """
        rev_id = self._get_revision_id()
        req = hq_pb2.GetInstanceRevRequest(id=instance_id, rev=rev_id)
        resp = self._retry(self._hq_stub.get_instance_rev, req)
        return resp.revision

    def report_status(self, instance_id, status):
        """
        :type instance_id: unicode
        :type status: types_pb2.RevisionStatus
        """
        status.id = self._get_revision_id()
        req = hq_pb2.ReportInstanceRevStatusRequest(instance_id=instance_id, status=status)
        self._hq_stub.report_instance_rev_status(req)

    def report_status_v2(self, instance_id, status):
        """
        :type instance_id: unicode
        :type status: types_pb2.RevisionStatus
        """
        service_id = instance_id.split("@")[1]
        status.id = self._get_revision_id()
        create_instance_if_not_exists = self._last_reported_conf_id != status.id
        req = hq_pb2.ReportInstanceRevStatusV2Request(
            instance_id=instance_id,
            service_id=service_id,
            status=status
        )
        if create_instance_if_not_exists:
            req.policies.append(hq_pb2.CREATE_INSTANCE_IF_NOT_EXISTS)
        self._hq_stub.report_instance_rev_status_v2(req)
        self._last_reported_conf_id = status.id

    def find_instances(self, service_id):
        req = hq_pb2.FindInstancesRequest()
        req.filter.service_id = service_id
        resp = self._retry(self._hq_stub.find_instances, req)
        return resp.instance
