# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import random
import logging

import gevent
from sepelib.gevent import greenthread

log = logging.getLogger('hq-state-reporter')


class IReportExecutor(object):
    def report(self, instance_id, status):
        """

        :type instance_id: six.text_type
        :type status: any
        """
        raise NotImplementedError


class ReportExecutorV1(IReportExecutor):
    def __init__(self, hq_client):
        self._hq_client = hq_client

    def report(self, instance_id, status):
        log.info('Reporting instance revision state to HQ')
        self._hq_client.report_status(instance_id, status)


class ReportExecutorV2(IReportExecutor):
    def __init__(self, hq_client):
        self._hq_client = hq_client

    def report(self, instance_id, status):
        log.info('Reporting instance revision state to HQ (V2)')
        self._hq_client.report_status_v2(instance_id, status)


class HqInstanceStatusReporter(greenthread.GreenThread):
    LOOP_DELAY = 1.0

    def __init__(self, report_executor, status_cacher, min_report_delay, max_report_delay, max_report_delay_jitter,
                 instance_id, enable_hq_report):
        """
        :type report_executor: IReportExecutor
        :type min_report_delay: int | float
        :type max_report_delay: int | float
        :type max_report_delay_jitter: int | float
        :type status_cacher: instancectl.status.cacher.InstanceRevisionStatusCacher
        :type instance_id: unicode
        :type enable_hq_report: bool
        """
        super(HqInstanceStatusReporter, self).__init__()
        self._report_executor = report_executor
        self._min_report_delay = min_report_delay
        self._max_report_delay = max_report_delay
        self._max_report_delay_jitter = max_report_delay_jitter
        self._status_cacher = status_cacher
        self._instance_id = instance_id
        self._enable_hq_report = enable_hq_report

    def _run(self):
        max_delay = self._get_periodic_report_delay()
        log.info('Waiting for instance status change (at most %.2f seconds)', max_delay)
        result = self._status_cacher.status_changed_event.wait(max_delay)
        if result:
            log.info('Instance status change triggered')
        else:
            log.info('Instance status change waiting timed out')
        self._status_cacher.status_changed_event.clear()
        self.report()

    def run(self):
        if not self._enable_hq_report:
            return
        while True:
            try:
                self._run()
            except Exception:
                log.exception('HQ instance status reporter thread fail')
            gevent.sleep(self._min_report_delay)

    def _get_periodic_report_delay(self):
        delay = self._max_report_delay + (random.random() - 0.5) * self._max_report_delay_jitter
        return max(delay, self._min_report_delay)

    def report(self):
        if not self._enable_hq_report:
            return
        status = self._status_cacher.get_status()
        try:
            self._report_executor.report(self._instance_id, status)
        except Exception:
            log.exception('Cannot report state to HQ')
